from typing import List

from common.backend.constants import (
    CONVERSATION_DEFAULT_NAME,
    DEFAULT_TITLE_GENERATION_ERROR,
    PROMPT_SEPARATOR_LENGTH,
)
from common.backend.models.base import MediaSummary
from common.backend.utils.dataiku_api import dataiku_api
from common.backend.utils.llm_utils import (
    get_alternative_llm,
    get_llm_completion,
    handle_response_trace,
)
from common.llm_assist.fallback import get_fallback_completion, is_fallback_enabled
from common.llm_assist.logging import logger
from common.solutions.prompts.conversation_title import MEDIA_QA_CONVERSATION_TITLE_PROMPT, TITLE_USER_PROMPT
from dataiku.langchain.dku_llm import DKULLM
from dataikuapi.dss.llm import DSSLLMCompletionResponse


class SummaryTitler():
    def __init__(self) -> None:
        self.webapp_config = dataiku_api.webapp_config

    def list_summaries(self, summaries: List[MediaSummary]) -> str:
        folder = dataiku_api.folder_handle
        all_summaries = ""
        if len(summaries) < 1:
            raise Exception("No media summaries found.")
        all_summaries += f"""
            {'-'*PROMPT_SEPARATOR_LENGTH} START OF SUMMARIES {'-'*PROMPT_SEPARATOR_LENGTH}
            """
        for media_summary in summaries:
            original_file_name = media_summary.get("original_file_name", "")
            topics = media_summary.get("topics", [])
            metadata_path = media_summary.get("metadata_path")
            if not metadata_path:
                logger.error(f"metadata_path is not provided for document {original_file_name}", log_conv_id=True)
                raise Exception(f"metadata_path is not provided for document {original_file_name}")
            extract_summary = folder.read_json(metadata_path)
            summary_text = extract_summary.get("summary", "")
            all_summaries += f"""
            File: {original_file_name}
            Topics: {topics}
            Summary: {summary_text}
            {'-'*PROMPT_SEPARATOR_LENGTH}
            """
        all_summaries += f"{'-'*PROMPT_SEPARATOR_LENGTH} END OF SUMMARIES {'-'*PROMPT_SEPARATOR_LENGTH}"
        return all_summaries

    def generate_summary_title(self, summaries: List[MediaSummary], first_attempt: bool = True) -> str:
        title_llm: DKULLM = get_alternative_llm("title_llm_id")
        media_summaries: str = self.list_summaries(summaries)
        system_prompt = MEDIA_QA_CONVERSATION_TITLE_PROMPT
        user_prompt = TITLE_USER_PROMPT.format(generated_content=media_summaries)
        conversation_title = CONVERSATION_DEFAULT_NAME
        try:
            completion = get_llm_completion(title_llm)
            if not first_attempt:
                completion = get_fallback_completion(completion)
            completion.with_message(system_prompt, role="system")
            completion.with_message(user_prompt, role="user")
            resp: DSSLLMCompletionResponse = completion.execute()
            handle_response_trace(resp)
            conversation_title = str(resp.text) if resp.text else CONVERSATION_DEFAULT_NAME
            if error_message := resp._raw.get("errorMessage"):
                logger.error(error_message, log_conv_id=True)
                raise Exception(error_message)
            logger.debug(f"conversation_title: {conversation_title}", log_conv_id=True)
            return conversation_title
        except Exception as e:
            logger.exception(f"{DEFAULT_TITLE_GENERATION_ERROR}: {e}.", log_conv_id=True)
            # checking if we can use a fallback LLM
            fallback_enabled = is_fallback_enabled(title_llm)
            if first_attempt and fallback_enabled:
                conversation_title = self.generate_summary_title(summaries, False)
            return conversation_title
