from functools import lru_cache
from typing import Dict, List, Tuple

import dataiku
import urllib3
from common.llm_assist.logging import logger
from dataikuapi.dss.project import DSSProject
from portal.backend.models import AnswersInfo
from portal.backend.utils.permissions_utils import user_has_access_to_project

# TEMPORARY
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)


ANSWERS_WEBAPP = "webapp_document-question-answering_document-intelligence-explorer"

@lru_cache(maxsize=128)
def _get_answers_info(selected_projects: Tuple[str, ...]) -> Dict[str, AnswersInfo]:
    client = dataiku.api_client()
    if not selected_projects:
        return {}
    answers_info: Dict[str, AnswersInfo] = {}
    for project_key in selected_projects:
        project: DSSProject = client.get_project(project_key)
        project_webapps = project.list_webapps()
        for webapp in project_webapps:
            if webapp.get("type") == ANSWERS_WEBAPP:
                webapp_obj = webapp.to_webapp()
                webapp_settings = webapp_obj.get_settings().get_raw()
                webapp_state = webapp_obj.get_state()
                if webapp_settings.get("config", {}).get("answers_description") and webapp_settings.get(
                    "config", {}
                ).get("enable_answers_api"):
                    answers_info[f"answers:{project_key}:{webapp.id}"] = {
                        "project_key": project.project_key,
                        "webapp_name": webapp.name,
                        "webapp_id": webapp.id,
                        "webapp_description": webapp_settings.get("config", {}).get("answers_description"),
                        "webapp_running": webapp_state.running,
                    }
    return answers_info

def get_answers_info(selected_projects) -> Dict[str, AnswersInfo]:
    return _get_answers_info(tuple(selected_projects))

@lru_cache(maxsize=128)
def accessible_webapps_workspaces(user_name: str) -> List[str]:
    client = dataiku.api_client()
    user = client.get_user(user_name)
    as_client = user.get_client_as() #Needs admin rights
    as_client._session.verify = False
    workspaces = as_client.list_workspaces(as_objects=True)
    webapps: List[str] = []
    for workspace in workspaces:
        for ob in workspace.list_objects():
            if ob.data and ob.data.get("reference"):
                if ob.data["reference"].get("type") == "WEB_APP":
                    project_key = ob.data["reference"].get("project_key")
                    id = ob.data["reference"].get("id")
                    webapps.append(f"{project_key}:{id}")
    return webapps


def filter_answers_per_user(user: str, answers: List[str]) -> List[str]:
    accessible_answers = []
    try:
        webapps = accessible_webapps_workspaces(user)
        projects_permissions: Dict[str, bool] = {}
        for answer in answers:
            project_key = answer.split(":")[1]
            if project_key in projects_permissions:
                if projects_permissions[project_key]:
                    accessible_answers.append(answer)
                continue
            webapp_id = answer.split(":")[2]
            if user_has_access_to_project(user, project_key):
                accessible_answers.append(answer)
                projects_permissions[project_key] = True
            else:
                # check if the answers is inside a workspace
                if f"{project_key}:{webapp_id}" in webapps:
                    logger.debug("User has access to a workspace that contains the webapp")
                    accessible_answers.append(answer)
                    projects_permissions[project_key] = True
                else:
                    projects_permissions[project_key] = False
    except Exception as e:
        logger.exception(f"Error filtering answers per user: {e}")
    return accessible_answers
