import unittest

from common.backend.utils.streaming_utils import TaggedTextStreamHandler


class TestTaggedTextStreamHandler(unittest.TestCase):

    def setUp(self):
        self.opening_tags = ["<citation"]
        self.closing_tags = ["</citation>"]
        self.handler = TaggedTextStreamHandler(self.opening_tags, self.closing_tags)

    def test_process_text_chunk_no_tags(self):
        text_chunk = "No tags here"
        processed_text = self.handler.process_text_chunk(text_chunk)
        self.assertEqual(processed_text, text_chunk)

    def test_process_text_chunk_with_tags(self):

        text_chunk = "<citation>Hello</citation>"
        processed_text = self.handler.process_text_chunk(text_chunk)
        self.assertEqual(processed_text, "<citation>Hello</citation>")

    def test_buffering_with_incomplete_tag(self):

        incomplete_chunk = "other text <cit"
        self.assertTrue(self.handler.ends_with_incomplete_opening_tag(incomplete_chunk))
        self.handler.process_text_chunk(incomplete_chunk)
        self.assertTrue(self.handler.buffer_text)

    def test_buffering_working(self):
        incomplete_chunk = "other text <cit"
        self.assertTrue(self.handler.ends_with_incomplete_opening_tag(incomplete_chunk))
        processed_text_chunk = self.handler.process_text_chunk(incomplete_chunk)
        self.assertTrue(self.handler.buffer_text)
        self.assertEqual(processed_text_chunk, "")

        next_chunk = "ation>Hello</citation>"
        processed_text_chunk = self.handler.process_text_chunk(next_chunk)
        self.assertEqual(processed_text_chunk, "other text <citation>Hello</citation>")

    def test_two_split_citations(self):

        first_chunk = "other text <cit"
        processed_text_chunk = self.handler.process_text_chunk(first_chunk)
        self.assertTrue(self.handler.buffer_text)
        self.assertEqual(processed_text_chunk, "")

        second_chunk = "ation>Hello</cit"
        processed_text_chunk = self.handler.process_text_chunk(second_chunk)
        self.assertTrue(self.handler.buffer_text)
        self.assertEqual(processed_text_chunk, "")
        third_chunk = "ation>. Something else ..."
        processed_text_chunk = self.handler.process_text_chunk(third_chunk)
        self.assertEqual(processed_text_chunk, "other text <citation>Hello</citation>. Something else ...")

    def test_non_tags(self):
        non_tag_chunk = "some text <not a tag> some more text"
        self.assertFalse(self.handler.ends_with_incomplete_opening_tag(non_tag_chunk))
        processed_text_chunk = self.handler.process_text_chunk(non_tag_chunk)
        self.assertFalse(self.handler.buffer_text)
        self.assertEqual(processed_text_chunk, non_tag_chunk)

    def test_reset_buffering(self):
        self.handler.buffer_text = True
        self.handler.buffered_text = "<citation>buffered text</citation>"
        self.handler.reset_buffering()
        self.assertFalse(self.handler.buffer_text)
        self.assertEqual(self.handler.buffered_text, "")

    def test_buffer_contains_opening_tag(self):
        self.handler.buffered_text = "Some <citation> text"
        self.assertTrue(self.handler.buffer_contains_opening_tag())

    def test_buffer_contains_closing_tag(self):
        self.handler.buffered_text = "Some text </citation>"
        self.assertTrue(self.handler.buffer_contains_closing_tag())

