import { marked } from 'marked'
import dompurify from 'dompurify'
import Prism from 'prismjs'

type Options = {
  target: '_blank' | '_self' | '_parent' | '_top'
}

const renderer = new marked.Renderer()

/**
 * Render and sanitize a markdown string
 */
export function md(
  str: string,
  translations: any,
  queryIndex: number,
  options: Options = { target: '_blank' }
): string {
    // Add hook to ensure links have target and rel attributes
  dompurify.addHook('afterSanitizeAttributes', (node) => {
    if (node.tagName === 'A') {
      // Ensure every link has target="_blank"
      if (!node.getAttribute('target')) {
        node.setAttribute('target', '_blank')
      }

      // Set rel="noopener noreferrer" for security
      if (node.getAttribute('target') === '_blank') {
        node.setAttribute('rel', 'noopener noreferrer')
      }
    }
  })

  // Custom link renderer
  renderer.link = function (href, title, text) {
    const link = marked.Renderer.prototype.link.apply(this, [href, title, text]);
    return link.replace(
      '<a',
      `<a target="${options.target}" rel="noopener noreferrer" style="color: #2563EB; text-decoration: underline;"`
    );
  };


  // Paragraph renderer to preserve newlines as <br />
  //isok
  renderer.paragraph = function (text) {
    return `<p style="margin: 0.5rem 0;">${text.replace(/\n/g, '<br />')}</p>`;
  };

  // add from asw
  
  // List renderer for ordered and unordered lists
  renderer.list = function (body, ordered, start) {
    const type = ordered ? 'ol' : 'ul';
    const listStyle = ordered ? 'list-style-type: decimal;' : 'list-style-type: disc;';
    const paddingLeft = 'padding-left: 1.5rem;';
    const marginBottom = 'margin-bottom: 1rem;';
    const style = `${listStyle} ${paddingLeft} ${marginBottom}`;
    const startAttr = ordered && start !== 1 ? ` start="${start}"` : '';

    return `<${type} style="${style}"${startAttr}>${body}</${type}>`;
  };


  // List item renderer
  renderer.listitem = function (text) {
    return `<li style="margin-bottom: 0.25rem;">${text}</li>`;
  };

  // Blockquote renderer
  renderer.blockquote = function (quote) {
  return `
    <blockquote style="
      border-left: 2px solid #000;
      padding: 0.5rem;
      margin: 1rem 0;
      border-top-right-radius: 0.375rem;
      border-bottom-right-radius: 0.375rem;
      color: #374151;
      font-style: italic;
    ">
      ${quote}
    </blockquote>
  `;
};

  renderer.codespan = function (text) {
  return `<code style="
    background-color: #E5E7EB;
    color: #1F2937;
    padding: 0.25rem;
    border-radius: 0.125rem;
    font-family: monospace;
    font-size: 0.875rem;
  ">${text}</code>`;
};

  // Table renderer
  renderer.table = function (header, body) {
    return `
    <table style="
      border-collapse: collapse;
      border: 1px solid #D1D5DB;
      margin: 1rem 0;
      width: 100%;
    ">
      ${header}
      ${body}
    </table>
  `;
  };

  // Table row renderer
  renderer.tablerow = function (content) {
    return `<tr>${content}</tr>`;
  }

  // Table cell renderer
renderer.tablecell = function (content, flags) {
  const tag = flags.header ? 'th' : 'td';
  const fontWeight = flags.header ? 'font-weight: 600;' : '';
  const padding = 'padding: 0.5rem;';
  const border = 'border: 1px solid #D1D5DB;';
  const textAlign = flags.align ? `text-align: ${flags.align};` : 'text-align: left;';
  
  const style = `${fontWeight} ${padding} ${border} ${textAlign}`;
  
  return `<${tag} style="${style}">${content}</${tag}>`;
}

  // add from asw

  renderer.code = function (code, language) {
    const detectedLang = language && Prism.languages[language] ? language : 'plaintext'
    const highlightedCode = Prism.highlight(code, Prism.languages[detectedLang], detectedLang)
    return `
		<div class="code-wrapper">
			<div class="header row justify-between items-center">
				<div>${language ?? ''}</div>
				<div class="row items-center q-gutter-x-xs copy">
					<q-icon>
						<svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 12 12" fill="none">
							<g opacity="0.7">
							<path
								d="M5.23877 1.02441C4.41034 1.02441 3.73877 1.69599 3.73877 2.52441V7.5166C3.73877 8.34503 4.41034 9.0166 5.23877 9.0166H8.76416C9.59259 9.0166 10.2642 8.34503 10.2642 7.5166V2.52441C10.2642 1.69599 9.59259 1.02441 8.76416 1.02441H5.23877ZM4.73877 2.52441C4.73877 2.24827 4.96263 2.02441 5.23877 2.02441H8.76416C9.0403 2.02441 9.26416 2.24827 9.26416 2.52441V7.5166C9.26416 7.79274 9.0403 8.0166 8.76416 8.0166H5.23877C4.96263 8.0166 4.73877 7.79274 4.73877 7.5166V2.52441Z"
								fill="white"
							/>
							<path
								d="M1.73584 4.52867C1.73584 3.71648 2.12315 2.99474 2.72314 2.53809V8.51695H2.73639C2.75812 9.32642 3.42113 9.97594 4.23584 9.97594H4.4491C4.49024 9.98109 4.53217 9.98375 4.57471 9.98375H8.50943C8.05298 10.5865 7.32953 10.9759 6.51514 10.9759H4.23584C2.85513 10.9759 1.73584 9.85665 1.73584 8.47594V4.52867Z"
								fill="white"
							/>
							</g>
						</svg>
					</q-icon>
				<div class="dds-caption-400 code-copy-btn" data-clipboard-text="${encodeURIComponent(
          code
        )}" onclick="dkuAnswers_copyToClipboard(this)">
        ${translations['copy_code']}
        </div>
				</div>
			</div>
			<pre class="language-${detectedLang}"><code>${highlightedCode}</code></pre>
		</div>
		`
  }

  // Horizontal rule renderer
  renderer.hr = function () {
    return `<hr class="margin: 1.5rem 0; border-color: #D1D5DB;">`
  }


  const strippedStr = str.replace(
  /(```[\s\S]*?```)|(`[^`\r\n]*`)|("(?:<([^>\s\/]+)([^>]*)>)")|<([^>\s\/]+)([^>]*)>/g,
  (
    fullMatch,
    fencedCode,  // ```…```
    inlineCode,  // `…`
    quotedWhole, // "<...>"
    quoteName,
    quoteRest,
    name,
    rest
  ) => {
    if (fencedCode || inlineCode) return fullMatch;

    // quoted "<...>" placeholder (used for PII)
    if (quotedWhole) {
      return `"&lt;${quoteName}${quoteRest}&gt;"`;
    }

    // unquoted "<...>"
    if (name && /^[a-z][a-z0-9-]*$/.test(name)) {
      return fullMatch; // potentially a real tag
    }
    return `&lt;${name}${rest}&gt;`;
  }
)

  // Patterns to extract citations
  const citationPattern = /<citation index="(\d+)"[\s\S]*?quote="([\s\S]*?)"><\/citation>/g
  const legacyCitationPattern =
    /\[source_index\](\d+)\[\/source_index\][\s\S]*?\[source_quote\]([\s\S]*?)\[\/source_quote\]/g
  let match
  const citations: any[] = []

  // Extract citations from markdown
  while ((match = citationPattern.exec(strippedStr)) !== null) {
    const index = match[1]
    const quote = match[2]?.replace(/\n/g, '<br />')
    citations.push({ index, quote })
  }

  // Extract citations for the second pattern
  while ((match = legacyCitationPattern.exec(strippedStr)) !== null) {
    const index = match[1]
    const quote = match[2]?.replace(/\n/g, '<br />')
    citations.push({ index, quote })
  }

  // Replace citations with placeholders to avoid markdown parsing
  const placeholderStr = strippedStr
    .replace(citationPattern, '<citation_placeholder>')
    .replace(legacyCitationPattern, '<citation_placeholder>')

  // Convert markdown to HTML
  const markdownContent = marked(placeholderStr, {
    renderer,
    headerIds: false,
    mangle: false
  })

  // Replace placeholders with citation popups
  let citIndex = 0
  const modifiedMarkdownContent = markdownContent.replace(/<citation_placeholder>/g, () => {
    const citation = citations[citIndex]
    const citationHtml = `
      <span class="citation-popup cursor-pointer" id="query-${queryIndex}-citation-${citIndex}" data-index="${citation.index}"
        onclick="dkuAnswers_handleCitationClick(${citIndex},${citation.index}, ${queryIndex})"
        onmouseover="dkuAnswers_showHideQuote('query-${queryIndex}-citation-${citIndex}', true)"
        onmouseleave="dkuAnswers_showHideQuote('query-${queryIndex}-citation-${citIndex}', false)">[${citation.index}]
        <span style="display:block" class="q-menu q-position-engine scroll citation-quote-popup row items-center">
          <span class="q-banner__content col bs-font-small-2-normal">
            <b>“</b><i>${citation.quote}</i><b>”</b>
          </span>
        </span>
      </span>
    `
    citIndex++
    return citationHtml
  })

  // Sanitize the final HTML
  return dompurify.sanitize(modifiedMarkdownContent, {
    ADD_ATTR: ['target', 'onclick', 'onmouseover', 'onmouseleave']
  })
}
