import { test as base, expect } from "@playwright/test";
import { execSync } from "child_process";
import path from "path";
import fs from "fs";

export const test = base.extend<
    { logTestInfo: void },
    { workerStorageState: string }
>({
    storageState: ({ workerStorageState }, use) => use(workerStorageState),

    // Authenticate once per worker with a worker-scoped fixture.
    workerStorageState: [
        async ({ browser }, use) => {
            // Use parallelIndex as a unique identifier for each worker.
            const workerIndex = test.info().parallelIndex;
            const authFile = path.resolve(
                test.info().project.outputDir,
                `.auth/user-${workerIndex}.json`
            );
            if (fs.existsSync(authFile)) {
                // Reuse existing authentication state if any.
                await use(authFile);
                return;
            }
            const isStaging = process.env.STAGING === "1";
            // Perform authentication steps. Replace these actions with your own.
            const url = isStaging
                ? process.env.E2E_LOGIN_URL
                : process.env.E2E_LOCAL_LOGIN_URL;
            // Assign different users per worker
            const user = isStaging
                ? {
                      username: `${process.env.E2E_USER_NAME}${
                          workerIndex > 0 ? workerIndex + 1 : ""
                      }`,
                      password: process.env.E2E_PASSWORD,
                  }
                : {
                      username: process.env.E2E_LOCAL_USER_NAME,
                      password: process.env.E2E_LOCAL_PASSWORD,
                  };

            if (!user || !user.username || !user.password) {
                throw new Error(
                    `Missing credentials for worker ${workerIndex}`
                );
            }
            console.log(
                `Worker ${workerIndex} is authenticating with ${user.username}`
            );
            // Important: make sure we authenticate in a clean environment by unsetting storage state.
            const page = await browser.newPage({ storageState: undefined });
            await page.goto(url!);
            await page.getByPlaceholder("Username").fill(user.username);
            await page.getByPlaceholder("Username").press("Tab");
            await page.getByPlaceholder("Password").fill(user.password);
            await page.getByPlaceholder("Password").press("Enter");
            // Wait until the page receives the cookies.
            // Sometimes login flow sets cookies in the process of several redirects.
            // Wait for the final URL to ensure that the cookies are actually set.
            const authUrl = isStaging
                ? process.env.E2E_AUTH_URL
                : process.env.E2E_LOCAL_AUTH_URL;
            expect(authUrl).toBeTruthy();
            // Just commented for now, if DSS behavior doesn't change in the next release we will want to 
            // remove authUrl from configuration or something in this idea.
            // await page.waitForURL(authUrl!);
            await page.waitForURL("**/public-webapps/TESTINTEGRATIONPORTALPLUGIN/**/new");

            // // Alternatively, you can wait until the page reaches a state where all cookies are set.

            // End of authentication steps.

            await page.context().storageState({ path: authFile });
            await page.close();
            await use(authFile);
        },
        { scope: "worker" },
    ],
    logTestInfo: async ({ page, browserName }, use) => {
        const workerIndex = base.info().parallelIndex;
        const testName = base.info().title;
        const browserVersion = await page.context().browser()?.version();
        console.log(
            `Worker ${workerIndex} is running: ${testName} on ${browserName} (v${browserVersion})\n`
        );
        console.log("--- System Information Before Test File ---\n");

        try {
            // Get CPU Usage (macOS)
            console.log(
                execSync("top -l 1 | grep 'CPU usage'").toString() + "\n"
            );

            // Get Memory Usage (macOS)
            console.log(
                execSync(
                    "vm_stat | awk 'NR==1 || /Pages free/ || /Pages active/ || /Pages inactive/'"
                ).toString() + "\n"
            );
        } catch (error) {
            console.log(`⚠️ Failed to fetch system info: ${error}\n`);
        }
        console.log("--- Running test ---\n");

        await use(); // Ensures Playwright continues execution
    },
});
