import types
from typing import Optional


class TaggedTextStreamHandler():
    """
    Handles streamed text to apply specific text processing transformation on text-parts that are contained
    within tags.
    """
    def __init__(self, opening_tags: list, closing_tags: list, text_processing_function: Optional[types.FunctionType] = None):
        """
        :param opening_tags: list: The opening tags to detect in the streamed text.
        :param closing_tags: list: The closing tags to detect in the streamed text.
        :param text_processing_function: types.FunctionType: A function to call that processes the text contained
            between the opening and the closing tags. It's first argument must be a string (the text) 
            and it's second argument must be a dictionary or parameters.
        """
        self.opening_tag_first_characters = [string[0] for string in opening_tags]
        self.min_opening_tag_length = min([len(opening_tag) for opening_tag in opening_tags])
        self.opening_tags = opening_tags
        self.closing_tags = closing_tags
        self.buffer_text = False
        self.buffered_text = ""
        self.text_processing_function = text_processing_function
        
    def process_text_chunk(self, text_chunk: str, text_processing_function_args: dict={}) -> str:
        """
        Processes a text chunk from the stream according to it's position relatively to the 'opening_tag' and 'closing_tag'
        :param text_chunk: str: The text chunk to process.
        :param text_processing_function_args: dict: The arguments to pass to the 'text_processing_function'.

        :return: processed_text_chunk: str: The result of the processing applied on the text chunk. 
            An empty string is returned if the text chunks should continue to be buffered.
        """

        processed_text_chunk = "" 
        if self.text_chunk_has_tag_opening_character(text_chunk) and not self.chunk_has_closing_tag(text_chunk):
            # Is the citation started but not finished in the same chunk?
            if self.ends_with_incomplete_opening_tag(text_chunk) or self.ends_with_incomplete_closing_tag(text_chunk):
                self.buffer_text = True
                self.buffered_text += text_chunk
            else:
                # Citation start tag is there (probably '<') but it is not an actual citation
                self.reset_buffering()
                return text_chunk
        else:
            if self.buffer_text:
                self.buffered_text += text_chunk
                if self.buffer_can_be_checked():
                    if not self.buffer_contains_opening_tag():
                        processed_text_chunk = self.buffered_text
                        self.reset_buffering()
                    if self.buffer_contains_closing_tag():
                        if self.text_processing_function:
                            self.buffered_text = self.text_processing_function(self.buffered_text,
                                                                               text_processing_function_args)
                        processed_text_chunk = self.buffered_text
                        self.reset_buffering()
            else:
                processed_text_chunk = text_chunk
        return processed_text_chunk 
    
    def text_chunk_has_tag_opening_character(self, text_chunk):
        result = False
        if text_chunk:
            for character in self.opening_tag_first_characters:
                if character in text_chunk:
                    result = True
        return result 

    def buffer_can_be_checked(self):
        """
        Informs if the buffer is big enough to be checked or not.
        """
        if len(self.buffered_text) > self.min_opening_tag_length:
            return True
        else:
            return False
    
    def buffer_contains_opening_tag(self):
        """
        Informs if the buffer contains an 'opening_tag' or not.
        """
        result = False
        for tag in self.opening_tags:
            if tag in self.buffered_text:
                result = True
        return result
    
    def buffer_contains_closing_tag(self):
        """
        Informs if the buffer contains a 'closing_tag' or not.
        """
        result = False
        for closing_tag in self.closing_tags:
            if closing_tag in self.buffered_text:
                result = True
        return result
    
    def reset_buffering(self):
        """
        Stops the text buffering and resets the buffered text.
        """
        self.buffer_text = False
        self.buffered_text = ""


    def chunk_has_closing_tag(self, text_chunk):
        """
        Checks if the chunk contains a complete closing tag
        """
        result = False
        for closing_tag in self.closing_tags:
            if closing_tag in text_chunk:
                result = True
        return result
    
    def ends_with_incomplete_closing_tag(self, text_chunk):
        """
        Checks if the buffering should still wait for a closing tag
        """
        result = False
        for tag in self.closing_tags:
            start_tag_parts = [tag[:i] for i in range(1, len(tag))][1:]
            if any(text_chunk.endswith(part) for part in start_tag_parts) or (tag in text_chunk):
                return True
        return result

    def ends_with_incomplete_opening_tag(self, text_chunk):
        """
        Checks if the buffering should still wait for a opening tag
        """
        result = False
        for tag in self.opening_tags:
            start_tag_parts = [tag[:i] for i in range(1, len(tag))][1:]
            if any(text_chunk.endswith(part) for part in start_tag_parts) or (tag in text_chunk):
                return True
        return result