from typing import Dict, Optional

import dataiku
from common.backend.constants import DEFAULT_MAX_LLM_TOKENS, DEFAULT_TEMPERATURE, LOWEST_TEMPERATURE
from common.backend.utils.dataiku_api import dataiku_api
from common.llm_assist.logging import logger
from dataikuapi.dss.llm import DSSLLMCompletionQuery
from portal.backend.constants import DKU_CALLER_SEC_TOKEN
from portal.backend.utils.answers_utils import get_answers_info
from portal.backend.utils.permissions_utils import get_user_display_name, get_user_email, get_user_groups


def get_llm_completion(llm_id: str, user: str, agent_project_key: Optional[str] = "", conversation_id: Optional[str]="") -> DSSLLMCompletionQuery:
    # Completion settings are not changed for 'Agents' and 'Answers'
    webapp_config: Dict[str, str] = dataiku_api.webapp_config
    completion: DSSLLMCompletionQuery
    if llm_id.startswith("answers:"):
        from portal.solutions.chatcompletion.answers import AnswersCompletionQuery

        project_id = llm_id.split(":")[1]
        webapp_id = llm_id.split(":")[2]
        answers_info = get_answers_info(webapp_config.get("projects_keys"))
        agent_name = answers_info[llm_id].get("webapp_name") if llm_id in answers_info else "No agent name provided"
        completion = AnswersCompletionQuery(llm_id, project_id, webapp_id, agent_name)
        logger.debug(f"completion settings for Answers '{llm_id}': {completion.settings}", log_conv_id=True)
    else:
        if not (llm_id.startswith("agent:") or llm_id.startswith("retrieval-augmented-llm:")):
            from common.backend.utils.config_utils import resolve_webapp_param

            llm = dataiku.api_client().get_default_project().get_llm(llm_id)
            completion = llm.new_completion()
            use_advanced_params: bool = webapp_config.get("show_advanced_settings", False)  # type: ignore
            temperature: Optional[float] = resolve_webapp_param(
                "llm_temperature", default_value=DEFAULT_TEMPERATURE, advanced_mode_enabled=use_advanced_params
            )
            completion.settings["maxOutputTokens"] = resolve_webapp_param(
                "max_llm_tokens", default_value=DEFAULT_MAX_LLM_TOKENS, advanced_mode_enabled=use_advanced_params
            )
            if (
                temperature is not None
            ):  # The temperature is passed as a completion parameter only if it has been resolved without a 'None' value
                if temperature > LOWEST_TEMPERATURE:
                    logger.warn(
                        f"The LLM '{llm_id}' temperature '{temperature}' is '>= {LOWEST_TEMPERATURE if temperature < 1.0 else 1.0}'.", log_conv_id=True
                    )
                completion.settings["temperature"] = temperature
            logger.debug(
                f"completion settings for LLM '{llm_id}' will be the ones of the 'Main LLM': {completion.settings}", log_conv_id=True
            )
        else:
            llm = dataiku.api_client().get_project(agent_project_key).get_llm(llm_id)
            completion = llm.new_completion()
            # For Agents we add security tokens
            logger.debug("Bypassing security check so no sec token is sent to agent", log_conv_id=True)
            user_groups = get_user_groups(user)  #No need for admin rights
            email = get_user_email(user)
            user_info = [f"dss_user_login:{user}"]
            if email:
                user_info.append(f"dss_user_emailaddress:{email}")
            
            if not conversation_id:
                logger.debug("The conversation_id was not part of the 'conversation_params' and can't be passed to the completion context.")
            completion.with_context(
                {
                    DKU_CALLER_SEC_TOKEN: user_groups + [f"dss_group:{group}" for group in user_groups] + user_info,
                    "dku_user_email": email,
                    "dku_user_login": user,
                    "dku_user_display_name": get_user_display_name(user),
                    "dku_conversation_id": conversation_id
                }
            )
            logger.debug(f"completion settings for Agent '{agent_project_key}:{llm_id}': {completion.settings}", log_conv_id=True)
    return completion
