from typing import Any, Dict, List

import dataiku
from common.llm_assist.logging import logger


def project_permissions(project_key: str) -> Dict[str, Any]:
    client = dataiku.api_client()
    project = client.get_project(project_key)
    permissions: Dict[str, Any] = project.get_permissions()
    return permissions

def get_user_groups(user: str) -> List[str]:
    client = dataiku.api_client()
    user_groups: List[str] = client.get_user(user).get_info().groups #No need to be admin, requires DSS14 
    logger.debug(f"User [{user}] groups: {user_groups}")
    return user_groups

def get_user_email(user: str) -> str:
    email = ""
    client = dataiku.api_client()
    try:
        email = client.get_user(user).get_info().email #No need for admin rights, requires DSS14
    except:
        logger.error(f"Failed to retrieve {user}'s email, proceeding..")
    return email

def get_user_display_name(user: str) -> str:
    display_name = ""
    client = dataiku.api_client()
    try:
        display_name = client.get_user(user).get_info().get_raw().get("displayName", "") #No need for admin rights, requires DSS14
    except:
        logger.error(f"Failed to retrieve {user}'s displayName, proceeding..")
    return display_name

def user_has_access_to_project(user: str, project_key: str) -> bool:
    permissions = project_permissions(project_key)
    client = dataiku.api_client()
    user_groups = get_user_groups(user) #No need to be admin
    logger.debug(f"User [{user}] groups: {user_groups}")
    logger.debug(f"Project [{project_key}] permissions: {permissions}")
    if user in permissions.get("owner", ""):
        logger.debug("User is the admin to the project")
        return True
    elif any(
        group
        in {
            permission.get("group")
            for permission in permissions.get("permissions", [])
            if permission.get("readProjectContent", False)
        }
        or client.get_group(group).get_definition().get("admin", False)
        for group in user_groups
    ):
        logger.debug("User is part of the group that has access to the project")
        # check if user is part of the groups that have access to the project
        return True
    elif user in {
        permission.get("user")
        for permission in permissions.get("permissions", [])
        if permission.get("readProjectContent", False) or permission.get("admin", False)
    }:
        logger.debug("User is part of the users that have access to the project")
        return True
    else:
        logger.debug("User does not have access to the project")
        return False
