from typing import Any, Dict, List, Union

from common.backend.models.source import AggregatedToolSources, Source
from common.backend.services.sources.sources_builder import (
    build_source,
)
from common.backend.services.sources.sources_formatter import sources_are_old_format
from common.backend.utils.json_utils import coerce_to_dict


def convert_to_rows_and_columns(columns: List[str], data: List[Any]):
    rows = [dict(zip(columns, row)) for row in data]

    columns_formatted = [{"name": col, "label": col, "field": col, "align": "left"} for col in columns]

    return {"rows": rows, "columns": columns_formatted}


def clean_images(images: List) -> List:
    return [coerce_to_dict(image) for image in images] if images else []


def process_agents_sources(input_sources: List[AggregatedToolSources]) -> List[AggregatedToolSources]: 
    output_sources = []
    for source in input_sources:
        generated_items: List[Source] = []
        items: List[Source] = source.get("items") or []

        for item in items:
            source_built = build_source(item)
            generated_items.append(source_built)
        output_sources.append(
            AggregatedToolSources(toolCallDescription=source.get("toolCallDescription"), items=generated_items)
        )
    return output_sources


def process_answers_sources(sources: Union[List[Dict], List[AggregatedToolSources]], used_tool: Dict) -> List[AggregatedToolSources]:
    if not sources_are_old_format(sources):
        return sources # type: ignore

    tool_name = ""
    tool_type = ""
    if used_tool:
        tool_name = used_tool.get("alias", "") or used_tool.get("name", "")
        tool_type = used_tool.get("type", "")
        if tool_type == "kb":
            tool_name = f"Knowledge Bank: {tool_name}"
        elif tool_type == "db":
            tool_name = f"Database: {tool_name}"
    items = [
        {
            "toolCallDescription": "Used " + tool_name,
            "items": [
                dict(
                    type="SIMPLE_DOCUMENT",
                    metadata=source.get("metadata", {}),
                    title=source.get("metadata", {}).get("source_title", ""),           #type: ignore
                    url=source.get("metadata", {}).get("source_url", ""),               #type: ignore
                    textSnippet=source.get("excerpt"),
                    images=clean_images((source.get("images", []))),                    #type: ignore
                )
                if tool_type == "kb"
                else dict(
                    type="RECORDS",
                    records=source.get("sample"),
                    generatedSqlQuery=used_tool.get("generatedSqlQuery", ""),
                    usedTables=used_tool.get("usedTables", []),
                )
                for source in sources
            ],
        }
    ]

    return items #type: ignore