import json
from typing import Any, Dict, List, Optional, Union

from common.backend.models.base import ConversationParams, MediaSummary, RetrievalSummaryJson
from common.backend.utils.llm_utils import handle_prompt_media_explanation
from common.backend.utils.prompt_utils import append_user_profile_to_prompt
from common.llm_assist.logging import logger
from common.solutions.chains.no_retrieval_chain import NoRetrievalChain
from dataiku.langchain.dku_llm import DKULLM
from portal.backend.constants import AGENT_ID, AUG_LLM_ID
from portal.backend.models import (
    AgentDetails,
    AgentGeneratedAnswer,
    AgentQuery,
    AgentsSelection,
    PortalLLMContext,
    PortalSource,
    ToolCall,
)
from portal.backend.utils.sources_utils import process_agents_sources, process_answers_sources


class QueryResolverChain(NoRetrievalChain):
    def __init__(
        self,
        llm: DKULLM,
        generated_queries_answers: List[AgentGeneratedAnswer],
        agents: List[AgentDetails],
        agents_queries: List[AgentQuery],
        query_builder_justification: str,
        chat_has_media: bool = False,
        agents_files_uploads: Optional[Dict[str, Dict[str, MediaSummary]]] = None,
        uploaded_files: Optional[List[MediaSummary]] = None,
    ):
        super().__init__(llm=llm, chat_has_media=chat_has_media)
        self.prompt_with_media_explanation = chat_has_media
        self.__act_like_prompt = """
        # Role and Guidelines
        You are an assistant that synthesizes information from different sources to provide a final answer to the user query.
        Your role is to read the initial user query and any answers generated by different agents to answer parts of the query or all and provide a final answer.
        
        Your responsibilities:
        - Read the initial user query and read answers from agents (based on sub-queries).
        - Understand which agent provided which answer (via `agent_name` and `agent_id`).
        - Each agent has its own scope and expertise. Use the answers provided by them as they are and rely on their answers to provide user with full answer. Do not make up your own.
        - Synthesize the information from the answers provided by the different agents to provide a final answer to the user query.
        - Do NOT alter the answers provided by external agents.
        - If you need to provide a different answer, make an additional one and clearly mention it's your own.

        Use the agent metadata to help you understand each agents's scope and its answers.
        """
        self.__system_prompt = f"""
        Given the initial user query, any possible generated answers, and the context of the conversation, provide a final answer to the initial user query.
        Don't change the answer generated by the agents. If you need to provide a different answer, provide it in a new answer and mention that you are providing a different answer.
        Here is additional metadata that might help you about the available external agents:
        {json.dumps(agents)}
        """
        self.agents = agents
        self.agents_files_uploads = agents_files_uploads
        self.agents_queries = agents_queries
        self.uploaded_files = uploaded_files
        self.generated_queries_answers = generated_queries_answers
        self.query_builder_justification = query_builder_justification

    @property
    def act_like_prompt(self) -> str:
        return self.__act_like_prompt

    @property
    def system_prompt(self) -> str:
        return self.__system_prompt


    def load_role_and_guidelines_prompts(self, params: ConversationParams):
        user_profile = params.get("user_profile", None)
        include_user_profile_in_prompt = bool(self.webapp_config.get("include_user_profile_in_prompt", False))
        self.__system_prompt = append_user_profile_to_prompt(
            system_prompt=self.__system_prompt,
            user_profile=user_profile,
            include_full_user_profile=include_user_profile_in_prompt,
        )
        self.__system_prompt = handle_prompt_media_explanation(
            system_prompt=self.__system_prompt, has_media=self.prompt_with_media_explanation
        )

    def get_computed_system_prompt(self, params: ConversationParams) -> str:
        gen_queries_answers_json = (
            [
                json.dumps(
                        {
                            "sub_query": query_answer.get("sub_query", "no query provided"),
                            "agent_answer": query_answer.get("agent_answer", "no answer provided"),
                            "agent_id": query_answer.get("agent_id", ""),
                            "agent_name": query_answer.get("agent_name", ""),
                        }
                )
                .replace("{", "{{")
                .replace("}", "}}")
                for query_answer in self.generated_queries_answers
            ]
            if self.generated_queries_answers
            else []
        )
        return r"""{act_like_prompt}
         
         {system_prompt}
         Generated sub queries and answers by agents based on user query:
         {generated_queries_answers}
         """.format(
            act_like_prompt=self.act_like_prompt,
            system_prompt=self.system_prompt,
            generated_queries_answers=gen_queries_answers_json,
        )


    def finalize_non_streaming(
        self,
        params: ConversationParams,
        answer_context: Union[str, Dict[str, Any], List[str]],
    ) -> RetrievalSummaryJson:
        return self._get_as_json(
            answer_context=answer_context,
            uploaded_docs=self.uploaded_files,
            user_profile=params.get("user_profile", None),
            sources=self.format_sources(self.generated_queries_answers),
        )

    def finalize_streaming(
        self,
        params: ConversationParams,
        answer_context: Union[str, Dict[str, Any], List[str]],
    ) -> RetrievalSummaryJson:
        return self._get_as_json(
            answer_context=answer_context,
            uploaded_docs=self.uploaded_files,
            user_profile=params.get("user_profile", None),
            sources=self.format_sources(self.generated_queries_answers),
        )

    def format_sources(self, sources: List[AgentGeneratedAnswer]) -> List[PortalSource]:
        formatted_sources: List[PortalSource] = []
        logger.debug(f"Formatting sources")
        # TODO refactor with agents_orchestrator
        for source in sources:
            #TODO should we handle aug llm differently
            is_agent = source.get("agent_id", "").startswith(AGENT_ID) or source.get("agent_id", "").startswith(AUG_LLM_ID)
            items: List[ToolCall] = []
            if source.get("sources"):
                if is_agent:
                    items = process_agents_sources(source["sources"]) #type: ignore
                else:
                    used_tool = source.get("used_tool")
                    items = process_answers_sources(source["sources"], used_tool) #type: ignore
            formatted_sources.append(
                {
                    "name": source["agent_name"] or "No agent name provided",
                    "id": source["agent_id"],
                    "type": "agent" if is_agent else "answers webapp",
                    "items": items,
                    "answer": source.get("agent_answer", ""),
                }
            )
        return formatted_sources

    def build_llm_context(
        self,
        user_profile: Optional[Dict[str, Any]] = None,
        uploaded_docs: Optional[List[MediaSummary]] = None,
    ) -> PortalLLMContext:
        llm_context: PortalLLMContext = {}
        if user_profile:
            llm_context["user_profile"] = user_profile
        if uploaded_docs:
            llm_context["uploaded_docs"] = [
                {
                    "original_file_name": uploaded_doc.get("original_file_name"),
                    "metadata_path": uploaded_doc.get("metadata_path", ""),
                }
                for uploaded_doc in uploaded_docs
            ]
        if self.agents_files_uploads:
            llm_context["agents_files_uploads"] = self.agents_files_uploads
        llm_context["agents_selection"] = AgentsSelection(
            calls=[
                AgentQuery(
                    query=agent_query_answer.get("sub_query", "no query provided"),
                    agent_id=agent_query_answer.get("agent_id", "no agent id provided"),
                )
                for agent_query_answer in self.generated_queries_answers
            ],
            justification=self.query_builder_justification,
        )
        return llm_context

    def _get_as_json(
        self,
        answer_context: Union[str, Dict[str, Any], List[str]],
        user_profile: Optional[Dict[str, Any]] = None,
        uploaded_docs: Optional[List[MediaSummary]] = None,
        sources: Optional[List[PortalSource]] = [],
    ) -> RetrievalSummaryJson:
        resp: RetrievalSummaryJson = {}
        answer = ""
        llm_context: PortalLLMContext = self.build_llm_context(user_profile=user_profile, uploaded_docs=uploaded_docs)
        gen_user_profile = user_profile
        if isinstance(answer_context, str):
            answer = answer_context
        if isinstance(answer_context, dict):
            answer = answer_context.get("answer", "")
            gen_user_profile = answer_context.get("user_profile", user_profile)
            if answer_context.get("images", None):
                resp["generated_images"] = answer_context["images"]
        else:
            logger.error(f"Generated answer is not a dictionary: {answer_context}")
            return resp
        resp.update(
            {
                "answer": answer,
                "sources": sources or [],
                "filters": None,
                "llm_context": llm_context,
                "user_profile": gen_user_profile or {},
            }
        )
        return resp
