import logging
import os

from dataiku.customwebapp import get_webapp_config


class LazyLogger:
    _logger = None
    _initialized = False

    @classmethod
    def _initialize_logger(cls):
        if not cls._initialized:
            try:
                webapp_config = get_webapp_config()
                log_level = webapp_config.get('log_level', 'INFO')
            except Exception as e:
                log_level = 'INFO'

            level = getattr(logging, log_level.upper(), logging.INFO)
            if not isinstance(level, int):
                raise ValueError(f'Invalid log level: {log_level}')

            if cls._logger is None:
                cls._logger = logging.getLogger(__name__)
            cls._logger.setLevel(level)

            if not cls._logger.handlers:
                formatter = logging.Formatter("%(asctime)s - %(name)s - [%(threadName)s (%(thread)d)] - %(levelname)s - %(message)s")

                debug_run_folder_path = os.getenv("DEBUG_RUN_FOLDER")
                if debug_run_folder_path:
                    local_logs_path = os.path.join(debug_run_folder_path, "logs", "answers.log")
                    os.makedirs(os.path.dirname(local_logs_path), exist_ok=True)
                    file_handler = logging.FileHandler(local_logs_path)
                    file_handler.setFormatter(formatter)
                    cls._logger.addHandler(file_handler)

                handler = logging.StreamHandler()
                handler.setFormatter(formatter)
                cls._logger.addHandler(handler)
                cls._logger.propagate = False

            cls._initialized = True
    
    def _log_conv_id(self, msg: str) :
        from common.backend.utils.context_utils import get_conv_id
        conv_id = get_conv_id()
        if not conv_id:
            return msg
        else:
            return f"{msg} (conv_id:'{conv_id}')"

    def debug(self, msg, log_conv_id: bool=False, *args, **kwargs):
        self._initialize_logger()
        if log_conv_id:
            msg = self._log_conv_id(msg)
        self._logger.debug(msg, *args, **kwargs) # type: ignore

    def info(self, msg, log_conv_id: bool=False, *args, **kwargs):
        self._initialize_logger()
        if log_conv_id:
            msg = self._log_conv_id(msg)
        self._logger.info(msg, *args, **kwargs) # type: ignore

    def warn(self, msg, log_conv_id: bool=False, *args, **kwargs):
        self._initialize_logger()
        if log_conv_id:
            msg = self._log_conv_id(msg)
        self._logger.warning(msg, *args, **kwargs) # type: ignore

    def error(self, msg, log_conv_id: bool=False, *args, **kwargs):
        self._initialize_logger()
        if log_conv_id:
            msg = self._log_conv_id(msg)
        self._logger.error(msg, *args, **kwargs) # type: ignore

    def critical(self, msg, log_conv_id: bool=False, *args, **kwargs):
        self._initialize_logger()
        if log_conv_id:
            msg = self._log_conv_id(msg)
        self._logger.critical(msg, *args, **kwargs) # type: ignore

    def exception(self, msg, log_conv_id: bool=False, *args, **kwargs):
        self._initialize_logger()
        if log_conv_id:
            msg = self._log_conv_id(msg)
        self._logger.exception(msg, *args, **kwargs) # type: ignore


logger = LazyLogger()
