from typing import Any, Dict, List, Union

from common.backend.constants import DOCUMENT_EXTENSIONS, IMAGE_EXTENSIONS, MAX_N_UPLOAD_FILES, MAX_UPLOAD_SIZE_MB
from common.backend.utils.config_utils import (
    SUPPORTED_LANGUAGES,
    format_feedback_choices,
    get_current_user_profile,
    get_user,
)
from common.backend.utils.dataiku_api import dataiku_api
from common.backend.utils.llm_utils import get_llm_capabilities
from common.llm_assist.logging import logger
from flask import Blueprint, Response, request
from portal.backend.db.user_profile import user_profile_sql_manager
from portal.backend.routes.utils import return_ko, return_ok

config_blueprint = Blueprint("config", __name__, url_prefix="/config")


@config_blueprint.route("/get_ui_setup", methods=["GET"])
def get_ui_setup() -> Response:
    """
    Fetches the configuration settings for UI setup from Dataiku and returns them.

    Returns:
        Response: A Flask response object containing the UI setup data.
    """
    config: Dict[str, Any] = dataiku_api.webapp_config
    examples: Union[str, List[str]] = config.get("example_questions", [])
    title: Union[str, None] = config.get("web_app_title")
    subtitle: Union[str, None] = config.get("web_app_subheading")
    lang: str = config.get("language", "en")
    placeholder: str = config.get("web_app_input_placeholder", "")
    feedback_positive_choices: List[str] = format_feedback_choices(config.get("feedback_positive_choices", []))
    feedback_negative_choices: List[str] = format_feedback_choices(config.get("feedback_negative_choices", []))
    llm_caps = get_llm_capabilities()
    if llm_caps.get("image_generation") and not config.get("upload_folder"):
        raise ValueError(
            "Image generation requires an upload folder to be set. Please configure one in the webapp settings and restart."
        )
    headers = dict(request.headers)
    disclaimer = config.get("disclaimer", "")
    user_profile = get_current_user_profile(headers, user_profile_sql_manager)
    result = {
        "examples": examples,
        "title": title,
        "subtitle": subtitle,
        "language": lang,
        "default_user_language": config.get("default_user_language", "English"),
        "user_settings": config.get("user_profile_settings", []),
        "current_user_profile": user_profile,
        "supported_languages": SUPPORTED_LANGUAGES,
        "input_placeholder": placeholder,
        "project": dataiku_api.default_project_key,
        "upload_folder_id": config.get("upload_folder"),
        "feedback_negative_choices": feedback_negative_choices,
        "feedback_positive_choices": feedback_positive_choices,
        "llm_capabilities": llm_caps,
        "llm_id": config.get("llm_id", ""),
        "image_generation_llm_id": config.get("image_generation_llm_id", "")
        if config.get("enable_image_generation", False)
        else "",
        "user": get_user(headers),
        "allow_general_feedback": config.get("allow_general_feedback", False),
        "disclaimer": disclaimer,
        "use_custom_rebranding": config.get("use_custom_rebranding", False),
        "custom_theme_name": config.get("custom_theme_name", ""),
        "custom_logo_file_name": config.get("custom_logo_file_name", ""),
        "custom_icon_file_name": config.get("custom_icon_file_name", ""),
        "max_upload_size_mb": config.get("max_upload_size_mb", MAX_UPLOAD_SIZE_MB),
        "max_n_upload_files": config.get("max_n_upload_files", MAX_N_UPLOAD_FILES),
        "image_extensions": list(IMAGE_EXTENSIONS),
        "document_extensions": list(DOCUMENT_EXTENSIONS),
        "max_images_per_user_per_week": config.get("max_images_per_user_per_week", 0),
        "display_agents_answers": config.get("display_agents_answers", False),
        "display_source_chunks": config.get("display_source_chunks", True),
        "display_sql_query": config.get("display_sql_query", False),
    }
    return return_ok(data=result)  # type: ignore


@config_blueprint.route("/user/profile", methods=["POST"])
def update_user_profile() -> str:
    headers = dict(request.headers)
    user = get_user(headers)
    settings = request.get_json()
    new_profile = settings.get("profile")
    try:
        profile = user_profile_sql_manager.get_user_profile(user)
        if profile:
            result = user_profile_sql_manager.update_user_profile(user=user, profile=new_profile)
        else:
            result = user_profile_sql_manager.add_user_profile(user=user, profile=new_profile)
    except Exception as e:
        logger.exception(f"Error updating user profile: {str(e)}")
        return return_ko(message="Error updating user profile")
    return return_ok(data=result)
