import { ServerApi } from '@/api/server_api'
import {
  ResponseStatus,
  type QuestionData,
  type Feedback,
  type AnswerRequest,
  type RetrievalType,
  type AnswerResponse,
  type AnswerData,
  type MediaSummary,
  ConversationType,
  LLLMStep,
  RetrievalMode
} from '@/models'
import type { Conversation } from '@/models/conversation'
import { ref, computed, watch } from 'vue'
import type { Ref } from 'vue'
import { useI18n } from 'vue-i18n'
import { nanoid } from 'nanoid'
import { useRouter } from 'vue-router'
import { useConversations } from './useConversations'
import { createNotification } from '@/common/utils'
import { useSettings } from './useSettings'
import { WT1iser } from '@/common/wt1'
import { Loading } from 'quasar'

import { WebSocketApi, type WebSocketEventHandlers } from '@/api/websocket_api'
import { useUI } from './useUI'

const mediaConversationStartTag = '_START_MEDIA_QA_CONVERSATION_'
const initData: QuestionData = {
  id: '',
  query: '',
  answer: '',
  filters: {},
  sources: [],
  chain_type: '',
  feedback: null,
  step: LLLMStep.LOADING,
  generated_images: [],
  uploaded_docs: []
}
const currentData = ref<QuestionData>({ ...initData })
const currentConversation = ref<Conversation | null>(null)
const loadingQuestion = ref<boolean>(false)

export function useConversation(id: Ref<string | null>) {
  const { t } = useI18n()

  const { getUserConversations, conversations, updateConversation } =
    useConversations()
  const router = useRouter()

  const loading = ref<boolean>(false)
  const error = ref<any>(null)

  const errorState = ref<boolean>(false)

  const {
    filtersSelections,
    knowledgeBankSelection,
    dbSelection,
    retrieverSelected,
    userSettings,
    updateUserSettingsMediaInfo
  } = useSettings()
  const { setup } = useUI()
  const queryFilters = computed(() => {
    const result: Record<string, any[]> = {}
    if (knowledgeBankSelection.value) {
      for (const key in filtersSelections.value) {
        if (
          filtersSelections.value[key] &&
          filtersSelections.value[key].length > 0
        ) {
          result[key] = filtersSelections.value[key]
        }
      }
    }
    return Object.keys(result).length > 0 ? result : null
  })

  function createTmpId() {
    return `_${nanoid()}`
  }

  function handleErrorResponse(response: any) {
    WT1iser.error(
      'handleErrorResponse',
      currentConversation.value?.data.length || 0,
      response
    )
    console.debug('Error:', response)
    loadingQuestion.value = false
    errorState.value = true
    const errorAnswerTranslated = t('error_answer')
    const errorMsgTranslated = t('error_message')
    currentData.value = {
      ...currentData.value,
      answer: errorAnswerTranslated
    }
    updateCurrentConversation()
    currentData.value = { ...initData }
    if (response.status && response.status == ResponseStatus.KO) {
      createNotification('negative', response.message)
    } else {
      createNotification('negative', errorMsgTranslated)
    }
  }

  const isNew = computed(() => {
    return id.value == null || id.value.startsWith('_')
  })

  async function getConversation() {
    // only get conversation history when not new
    if (isNew.value) return
    loading.value = true
    try {
      if (!id.value) return
      const response = await ServerApi.getConversation(id.value)
      if (response && response.data) {
        currentConversation.value = response.data
        currentConversation.value.data = currentConversation.value.data.map(
          (val) => {
            return {
              ...val,
              kbOn: val.llm_kb_selection?.length
                ? val.llm_kb_selection?.length > 0
                : false,
              dbOn: val.llm_db_selection?.sql_retrieval_table_list
                ? true
                : false
            }
          }
        )
        currentConversation.value.conversation_type =
          response.data.conversation_type
      }
    } catch (e) {
      error.value = e
    } finally {
      loading.value = false
    }
  }

  async function reset(as_media_conv: boolean = false) {
    loading.value = false
    error.value = null
    currentData.value = { ...initData }
    currentConversation.value = null
    currentData.value.conversation_type = as_media_conv
      ? ConversationType.MEDIA_QA
      : ConversationType.GENERAL
    conversations.value.forEach((conv) => {
      conv.selected = false
    })
    await getConversation()
  }
  async function summarizeFiles(formData: FormData) {
    let msg = 'generic_upload_error'
    try {
      WT1iser.newMediaConversation()
      Loading.show({
        message: t('extracting_data'),
        boxClass: 'bg-grey-2 text-grey-9',
        spinnerColor: 'primary'
      })
      const response = await ServerApi.summaryUploadFile(formData)
      msg = response.message ?? 'generic_upload_error'
      if (response.status == ResponseStatus.OK) {
        if (!id.value) {
          // Start a new conversation with summary
          currentConversation.value = startTmpConv()
          currentConversation.value.conversation_type =
            ConversationType.MEDIA_QA
          currentConversation.value.media_summaries =
            response.data.media_summaries
        } else {
          // Add summary to existing conversation as uploaded docs
          currentData.value.uploaded_docs = response.data.media_summaries
        }
        currentData.value = {
          ...currentData.value,
          query: mediaConversationStartTag
        }
        sendQuestion()
        Loading.hide()
        createNotification('positive', t('upload_successful'))
      } else if (msg) {
        throw Error(msg)
      } else {
        throw Error('generic_upload_error')
      }
    } catch (error: any) {
      createNotification('negative', t(msg))
      Loading.hide()
      loadingQuestion.value = false
    }
  }
  function stopQuery() {
    console.debug('Stopping query', currentData.value)
    loadingQuestion.value = false
    WebSocketApi.sendLogQuery({
      ...currentData.value,
      conversation_id: isNew.value ? '' : id.value
    })
  }
  function updateCurrentConversation() {
    if (
      currentConversation.value?.data.length &&
      currentConversation.value?.data.length > 0
    ) {
      currentConversation.value.data[
        currentConversation.value?.data.length - 1
      ] = {
        ...currentData.value
      }
    }
  }
  function updateMessageId() {
    if (
      currentConversation.value?.data.length &&
      currentConversation.value?.data.length > 0
    ) {
      currentConversation.value.data[
        currentConversation.value?.data.length - 1
      ].id = currentData.value.id
    }
  }
  async function sendQuestion(formData?: FormData) {
    WT1iser.askQuestion(currentConversation.value?.data.length || 0)
    if (!currentData.value.query && formData) {
      // User uploaded files without a question
      return summarizeFiles(formData)
    } else if (!currentData.value.query) {
      return
    }
    loadingQuestion.value = true
    currentData.value = {
      ...initData,
      query: currentData.value.query,
      chain_type: currentData.value.chain_type,
      filters: queryFilters.value,
      uploaded_docs: currentData.value.uploaded_docs
    }
    if (formData) {
      try {
        const response = await ServerApi.uploadFile(formData)
        if (response.status == ResponseStatus.OK) {
          currentData.value.uploaded_docs = response.data.media_summaries
        } else {
          throw Error('generic_upload_error')
        }
      } catch (error) {
        console.error('Error uploading files:', error)
        createNotification('negative', t('generic_upload_error'))
        loadingQuestion.value = false
        return
      }
    }
    if (isNew.value) {
      router.push({
        path: `/conversation/${createTmpId()}`
      })
    }
    try {
      fetchAnswer()
    } catch (e) {
      handleErrorResponse(e)
    }
  }
  async function handleQueryResponse(data: any) {
    console.debug('Query response:', data)
    try {
      const resp = JSON.parse(data)
      if (resp.status === ResponseStatus.OK) {
        if (loadingQuestion.value) {
          if (resp.data.step) {
            currentData.value.step = resp.data.step
          } else {
            const llm_context = resp.data.llm_context || {}
            currentData.value = {
              ...currentData.value,
              answer: currentData.value.answer + resp.data.answer,
              sources: resp.data.sources ?? [],
              filters: currentData.value.filters
                ? currentData.value.filters
                : resp.data.filters ?? null,
              kbOn: 'llm_kb_selection' in llm_context,
              dbOn: 'dataset_context' in llm_context,
              agentsOn: 'agents_selection' in llm_context,
              generated_images: resp.data.generated_images ?? [],
            }
          }
          updateCurrentConversation()
        }
      } else {
        handleErrorResponse(resp)
        if (isNew.value) {
          // trigger a list conv refresh
          await getUserConversations()
          if (conversations.value.length > 0) {
            await router.push({
              path: `/conversation/${conversations.value[0].id}`
            })
          }
        }
      }
    } catch (e) {
      console.error('Error:', e)
    }
  }
  function handlerWSDisconnect() {
    loadingQuestion.value = false
  }

  async function handleLogQueryComplete(response: any) {
    const resp: AnswerResponse = JSON.parse(response)
    console.debug('Query complete event received', resp)
    if (resp.status == ResponseStatus.OK) {
      currentData.value.id = resp.data.record_id ? resp.data.record_id.toString() : '0'
      const isNewConv = isNew.value
      if (isNewConv) {
        const data = resp.data as AnswerData
        if (currentConversation.value) {
          updateConversation(currentConversation.value.id, {
            name: data.conversation_infos.name,
            id: data.conversation_infos.id
          })
        }
      }
      // update user profile media info
      if (resp.data.generated_images && resp.data.generated_images.length > 0) {
        updateUserSettingsMediaInfo(resp.data.user_profile)
        setup.value.currentUserProfile = userSettings.value
      }
      loadingQuestion.value = false
      if (resp.data.generated_images && resp.data.generated_images.length > 0) {
        // Workaround here as for portal we only need to update current conv
        // when images are generated because thats when we recieve the generated images
        // Otherwise all the other data except for message id is updated in the handleQueryResponse
        updateCurrentConversation()
      }else{
        //So here we only update the id
        updateMessageId()
      }
      currentData.value = { ...initData }
      if (isNewConv) {
        // trigger a conv refresh
        await router.push({
          path: `/conversation/${resp.data.conversation_infos.id}`
        })
      }
    } else {
      handleErrorResponse(resp)
      if (isNew.value) {
        // trigger a list conv refresh
        await getUserConversations()
        await router.push({ path: `/new` })
      }
    }
  }
  function fetchAnswer() {
    loadingQuestion.value = true
    if (isNew.value) {
      if (!currentConversation.value?.conversation_type) {
        currentConversation.value = startTmpConv()
      }
      conversations.value = [
        { ...currentConversation.value },
        ...conversations.value
      ]
    }
    const question = currentData.value.query
    currentData.value.query =
      currentData.value.query === mediaConversationStartTag
        ? ''
        : currentData.value.query
    currentConversation.value?.data.push(currentData.value)
    const conversationType =
      currentConversation.value?.conversation_type ?? ConversationType.GENERAL

    const extractFields = (
      items: MediaSummary[] | undefined | null,
      fields: string[]
    ) =>
      (items || []).map((item) =>
        Object.fromEntries(
          fields.map((field) => [field, item[field as keyof MediaSummary]])
        )
      )
    let uploadedDocs = []
    if (
      conversationType == ConversationType.MEDIA_QA &&
      currentConversation.value &&
      isNew.value &&
      currentData.value.query === ''
    ) {
      // Only at initial summarization step
      console.debug('First Media Summary')
      uploadedDocs = extractFields(currentConversation.value.media_summaries, [
        'topics',
        'chain_type',
        'original_file_name',
        'file_path',
        'metadata_path',
        'token_estimate'
      ])
    } else {
      console.debug('Uploaded Docs')
      uploadedDocs = extractFields(currentData.value?.uploaded_docs, [
        'chain_type',
        'original_file_name',
        'file_path',
        'metadata_path'
      ])
    }

    const query_index_: number = currentConversation.value
      ? currentConversation.value.data.length - 1
      : 0
    const retrievalType: RetrievalType[] = retrieverSelected.value
      ? [
          ...(knowledgeBankSelection.value
            ? [{ type: RetrievalMode.KB, source: knowledgeBankSelection.value }]
            : []),
          ...(dbSelection.value
            ? [{ type: RetrievalMode.DB, source: dbSelection.value }]
            : [])
        ]
      : []
    const queryData: AnswerRequest = {
      conversation_id: id.value,
      conversation_type: conversationType,
      query: question,
      query_index: query_index_,
      filters: queryFilters.value,
      // chain_type: chainType,
      media_summaries: uploadedDocs,
      retrieval_enabled: retrieverSelected.value ?? false,
      retrieval_selection: retrievalType,
      user_profile: userSettings.value,
      app_id: setup.value.webAppId
    }
    try {
      const handlers: WebSocketEventHandlers = {
        onMessage: handleQueryResponse,
        onError: (data) => handleErrorResponse(JSON.parse(data)),
        onDisconnect: handlerWSDisconnect,
        onLogQueryComplete: handleLogQueryComplete
      }
      WebSocketApi.setupHandlers(handlers)
      WebSocketApi.sendMessage(queryData)
    } catch (e) {
      console.debug('Error:', e)
      handleErrorResponse(e)
    }
  }

  function startTmpConv() {
    const conversation: Conversation = {
      auth_identifier: '',
      id: createTmpId(),
      data: [],
      name: '',
      timestamp: Date.now() / 1000,
      selected: true,
      conversation_type: ConversationType.GENERAL,
      media_summaries: []
    }
    return conversation
  }

  async function clearHistory() {
    WT1iser.clearHistory(currentConversation.value?.data.length || 0)
    try {
      if (!id.value || isNew.value) return
      await ServerApi.clearHistory(id.value)
      if (currentConversation.value?.data) {
        currentConversation.value.data = []
      }
    } catch (e) {
      createNotification('negative', t('error_message'))
    }
  }

  async function logFeedback(
    record_id: string | number,
    item: QuestionData,
    feedback: Feedback
  ) {
    WT1iser.giveFeedback(
      currentConversation.value?.data.length || 0,
      feedback.value,
      feedback.message ? true : false
    )
    if (!id.value || isNew.value) return
    try {
      if(record_id === '') throw new Error('Record id is empty')
      item.feedback = feedback
      await ServerApi.logFeedback(id.value, record_id, item)

      if (currentConversation.value?.data) {
        currentConversation.value.data = currentConversation.value.data.map(
          (val, index) => {
            if (index === record_id) {
              const result = { ...val, feedback: feedback }
              return result
            }
            return val
          }
        )
      }
      createNotification('positive', t('feedback_success_message'))
    } catch (e) {
      createNotification('negative', t('error_message'))
    }
  }
  async function deleteFile(file: MediaSummary) {
    if (!id.value || isNew.value) return
    if (!file.file_path) return
    try {
      await ServerApi.deleteFile([file.file_path], [file.metadata_path ?? ''])
      if (currentConversation.value?.media_summaries) {
        currentConversation.value.media_summaries =
          currentConversation.value.media_summaries.map((doc) => {
            if (doc.file_path !== file.file_path) {
              return doc
            } else {
              return { ...doc, is_deleted: true, questions: [] }
            }
          })
      }
      currentConversation.value?.data.forEach((val) => {
        if (val.uploaded_docs) {
          val.uploaded_docs = val.uploaded_docs.map((doc) => {
            if (doc.file_path !== file.file_path) {
              return doc
            } else {
              return { ...doc, is_deleted: true, preview: '', questions: [] }
            }
          })
        }
      })
      createNotification('positive', t('file_delete_successful'))
    } catch (e) {
      createNotification('negative', t('file_delete_failed'))
    }
  }
  watch(
    () => id.value,
    (newVal, oldVal) => {
      if (
        !(
          id.value == null ||
          id.value.startsWith('_') ||
          oldVal?.startsWith('_')
        )
      )
        reset()
    },
    { immediate: true }
  )

  return {
    currentData,
    currentConversation,
    sendQuestion,
    stopQuery,
    loadingQuestion,
    errorState,
    reset,
    isNew,
    clearHistory,
    logFeedback,
    summarizeFiles,
    deleteFile
  }
}
