from datetime import datetime, timezone
from typing import Optional

from common.backend.constants import (
    BIGQUERY_DATETIME_FORMAT,
    BIGQUERY_STRING_DATE_FORMAT,
    LOGGING_DATES_FORMAT,
    SNOWFLAKE_DATETIME_FORMAT,
)


def from_datetime_to_dss_string_date(datetime: datetime) -> str:
    """
    Converts a datetime object into a dataiku DSS string datetime value.

    :param: datetime:  datetime.datetime: The datetime date to convert.

    :returns: dss_string_date: str: DSS string date associated with the 'datetime_date'.
    """
    # We remove the end of the string to get 'milliseconds' resolution instead of 'microseconds'.
    datetime_info = datetime.strftime("%Y-%m-%dT%H:%M:%S.%f")[:-3]
    timezone_info = datetime.strftime("%z")
    dss_string_date = datetime_info + timezone_info
    return dss_string_date


def get_string_date_interpreted_as_datetime(sql_dialect: Optional[str]=None)->str:
    """
    Computes a string date that is interpreted with a 'Datetime' storage datatype by a database
    """
    if sql_dialect == "Snowflake":
        return datetime.now(timezone.utc).strftime(SNOWFLAKE_DATETIME_FORMAT)
    elif sql_dialect == "BigQuery":
        return datetime.now(timezone.utc).strftime(BIGQUERY_DATETIME_FORMAT)
    else:
        return from_datetime_to_dss_string_date(datetime.now(timezone.utc))


def get_string_date_interpreted_as_string(sql_dialect: Optional[str]=None)->str:
    """
    Computes a string date that is interpreted with a 'Date-like' storage datatype by a database
    """
    if sql_dialect == "BigQuery":
        return datetime.now().strftime(BIGQUERY_STRING_DATE_FORMAT)
    # 'Snowflake' dialect is covered under LOGGING_DATES_FORMAT
    return datetime.now(timezone.utc).strftime(LOGGING_DATES_FORMAT)