import base64
import random
import string
import time
from typing import Any, Dict, List, Optional, Tuple

import dataiku
from common.backend.db.sql.tables_managers import GenericUserProfileSQL
from common.backend.models.base import (
    GeneratedMedia,
)
from common.backend.utils.dataiku_api import dataiku_api
from common.llm_assist.logging import logger
from werkzeug.utils import secure_filename


def handle_images(
    generated_images, user: str, user_profile: Optional[Dict[str, Any]], user_profile_sql_manager: GenericUserProfileSQL
) -> Tuple[List[GeneratedMedia], Optional[Dict[str, Any]]]:
    images: List[GeneratedMedia] = []
    config: Dict[str, str] = dataiku_api.webapp_config
    for image in generated_images:
        base64_data = image.get("file_data", "").split(",")[1]
        image_format = image.get("file_format", "")
        # Decode the Base64 data to bytes
        file_data = base64.b64decode(base64_data)
        # random file name
        random_str = "".join(random.choice(string.ascii_uppercase + string.digits) for _ in range(6))
        filename = secure_filename(f"{user}_{int(time.time())}_{random_str}.{image_format}")
        filename = f"{user}/{filename}"
        folder = dataiku.Folder(config.get("upload_folder"))
        with folder.get_writer(filename) as writer:
            writer.write(file_data)
        logger.debug("File uploaded to folder")
        images.append(
            {
                "file_path": filename,
                "file_format": image_format,
                "referred_file_path": image.get("referred_file_path", ""),
            }
        )
    if user_profile:
        user_profile = user_profile_sql_manager.update_generated_images_count_in_db(
            user, user_profile, len(images), config
        )
    else:
        logger.error("Couldn't update User profile in DB. No user profile was provided")
    return images, user_profile
