import json
from typing import Dict, List, Optional

from common.backend.models.base import (
    LlmHistory,
    MediaSummary,
)
from common.llm_assist.logging import logger


def extract_logging_uploaded_files_info(media_summaries: Optional[List[MediaSummary]]) -> List[Dict[str, str]]:
    return (
        [
            {
                "original_file_name": summary.get("original_file_name", ""),
                "metadata_path": summary.get("metadata_path", ""),
            }
            for summary in media_summaries
        ]
        if media_summaries
        else []
    )


def extract_uploaded_docs_from_history(history: List[LlmHistory]) -> List[MediaSummary]:
    """
    Extracts all uploaded documents + summaries from the history list and updates the list in place.
    This function modifies the original 'history' list by replacing certain elements with processed JSON strings where possible.
    """
    previous_media_summaries = []
    logger.debug(f"Extracting uploaded docs from history")
    for index, history_item in enumerate(history):
        for key in ["input", "output"]:
            try:
                # Attempt to parse the history_item[key] as JSON
                data = json.loads(history_item[key])  # type: ignore
                # Check if 'uploaded_docs' exists in the parsed JSON
                if "uploaded_docs" in data:
                    available_docs = [doc for doc in data["uploaded_docs"] if not doc.get("is_deleted", False)]
                    if available_docs:
                        previous_media_summaries.extend(available_docs)
                    # Replace the 'uploaded_docs' key with a JSON string containing only the necessary information
                    # So that we dont send all the extracted data to the llm as part of the history
                    history_json = {
                        "uploaded_docs": [
                            {
                                "original_file_name": doc["original_file_name"],
                                "file_path": doc["file_path"]
                                if doc.get("file_path") and not doc.get("is_deleted", False)
                                else "",
                                "metadata_path": doc.get("metadata_path", ""), #TODO: This is needed mostly for portal should we make it conditional?
                                "chain_type": doc.get("chain_type", ""),#TODO: This is needed mostly for portal should we make it conditional?
                                "is_file_deleted": doc.get("is_deleted", False),
                            }
                            for doc in data["uploaded_docs"]
                        ]
                    }
                    if "query" in data:
                        history_json["query"] = data["query"]
                    history[index][key] = json.dumps(history_json)  # type: ignore
            except (json.JSONDecodeError, TypeError, ValueError):
                # If JSON parsing fails or the input is not a string, skip to the next item
                continue
    return previous_media_summaries