from pathlib import Path
from typing import List, Optional

from common.backend.utils.file_extraction.pptx.pptx_utils import extract_pptx_slides_as_imgs
from common.backend.utils.file_utils import load_pdf_images_from_bytes
from common.llm_assist.logging import logger
from common.solutions.chains.generic_summary_chain import GenericSummaryChain
from dataikuapi.dss.llm import DSSLLMCompletionQuery


class DocAsImageSummaryChain(GenericSummaryChain):
    def __init__(self, file_data: bytes, original_file_name: str, language: Optional[str], file_path: str = "", b64_images: Optional[List[str]] = None):
        self._file_data = file_data
        self._original_file_name = original_file_name
        self._language = language
        self.file_path = file_path
        self.b64_images = b64_images

        self._completion = None

    @property
    def file_data(self) -> bytes:
        return self._file_data

    @property
    def original_file_name(self) -> str:
        return self._original_file_name

    @property
    def language(self) -> Optional[str]:
        return self._language


    def get_summary_completion(self) -> DSSLLMCompletionQuery:
        doc_as_image_system_prompt = """
        # Role and Guidelines
        Your role is to look at and summarize the pages of a document provided to you. These pages have been uploaded to a webapp.
        Your summary should mention the contents of each section of the document if possible.
        You should provide your answer in JSON format with the following keys
        - "summary": str: summary of all the pages provided to you
        - "topics": Array[str]: array of all the main topics that appear in the document provided.
        - "questions": Array[str]: array of 3 questions which can be asked based on the document provided to you.
        The JSON object should have double quotes for keys and values. It is important to follow the JSON format strictly.
        Provide a JSON object only. Do not add any other explanations, decorations, or additional information beyond the JSON object.
        """
        doc_as_image_user_prompt = """
        Summarize the following extracted document text
        Your JSON object:
        """
        if self.language:
            doc_as_image_system_prompt += f"""
        Your response should be in {self.language}
        """
        logger.debug(f"""
            Document as Image Summary system Prompt: {doc_as_image_system_prompt}
            Document as Image Summary user Prompt: {doc_as_image_user_prompt}
        """, log_conv_id=True)
        completion: DSSLLMCompletionQuery = self.init_completion()
        completion.with_message(doc_as_image_system_prompt, role="system")
        completion.with_message(doc_as_image_user_prompt, role="user")

        b64_images: List[str] = []
        extension = Path(self.original_file_name).suffix if self.original_file_name else ""

        if "pptx" in extension:
            if self.b64_images:
                b64_images = self.b64_images
            else:
                b64_images = extract_pptx_slides_as_imgs(self.file_path, self.file_data)
        else:
            b64_images = load_pdf_images_from_bytes(self.file_data)

        n_page: int = len(b64_images)
        msg = completion.new_multipart_message()
        for page, img_b64 in enumerate(b64_images):
            msg.with_text(f"{self.original_file_name}: Page {page + 1} of {n_page}").with_inline_image(img_b64)
        msg.add()
        return completion