import json
import locale
import os
from datetime import datetime
from logging.config import dictConfig
from pathlib import Path

from common.backend.utils.dataiku_api import dataiku_api

year, date = datetime.now().strftime("%Y"), datetime.now().strftime("%Y-%m-%d")
# Replace by your default project key that you are working on in dev
DEFAULT_PROJECT_KEY = "PORTAL"
LOGGING_DATASET = "conversations"
FEEDBACK_DATASET = ""
UPLOAD_FOLDER_ID = "upload_folder"
USER_PROFILE_DATASET = "user_profile"
LLM_ID = "openai:BSOpenAI:gpt-4o"
# TODO : Add dip_home to a .env file
DEFAULT_CONFIG = {
    # put your webapp desired config
    "webapp_config": {
        "primer_prompt": f"You are a general purpose assistant, be as concise as possible, The current Year is {year}. The current Date is {date} ",
        "system_prompt": f"""Please act as an assistant and comply to the following principles:
        Understand and Clarify the Task: Begin by comprehensively understanding the user's request. If necessary, ask clarifying questions to ensure accuracy in the response.""",
        "memory_token_limit": 8000,
        "web_app_title": "Ask directly questions to your document corpus",
        "web_app_subheading": "Answers are sourced from the processed documents, delivered in natural language and accompanied by the most relevant references.",
        "example_questions": [
            "What are the primary causes of the recent increase in global temperatures?",
            "How does quantum computing differ from classical computing in terms of processing information?",
            "What are the most used languages",
        ],
        "language": "en",
        "logging_dataset": LOGGING_DATASET,
        "feedback_positive_choices": [],
        "feedback_negative_choices": [],
        "allow_general_feedback": False,
        "general_feedback_dataset": FEEDBACK_DATASET,
        "web_app_input_placeholder": "Ask your question",
        "llm_id": LLM_ID,
        "permanent_delete": True,
        "upload_folder": UPLOAD_FOLDER_ID,
        "log_level": "DEBUG",
        "max_llm_tokens": 2048,
        "answers_ids": [
            "answers:DATAIKUANSWERS:s6EdGpV",
            "answers:DATAIKUANSWERS:w1mvOEo",
            "answers:DATAIKUANSWERS:wwds",
        ],
        "agents_ids": [],
        "agents_descriptions": [],
        "disclaimer": 'This application is powered by a Large Language Model. Review proposed outcomes in context with \u003cspan style\u003d"color: #267cfb; text-decoration: underline"\u003eResponsible AI\u003c/span\u003e considerations',
        "http_headers": [
            {"from": "Content-Security-Policy", "to": "frame-ancestors \u0027self\u0027;"},
            {"from": "Strict-Transport-Security", "to": "max-age\u003d31536000; includeSubDomains"},
            {"from": "X-Frame-Options", "to": "SAMEORIGIN"},
            {"from": "X-Content-Type-Options", "to": "nosniff"},
        ],
        "use_custom_rebranding": False,
        "custom_theme_name": "",
        "custom_logo_file_name": "",
        "custom_icon_file_name": "",
        "user_profile_settings": [
            {"from": "user_id", "to": "user_id"},
            {"from": "country", "to": "country"},
            {"from": "department", "to": "department"},
        ],
        "include_user_profile_in_prompt": True,
        "user_profile_dataset": USER_PROFILE_DATASET,
        "max_n_upload_files": 5,
        "max_upload_size_mb": 10,
        "allow_doc_as_image": True,
        "docs_per_page_as_image": 5,
        "enable_image_generation": False,
        "image_generation_llm_id": "",
        "image_generation_system_prompt": "",
        "max_images_per_user_per_week":5,

    },
    "default_project_key": DEFAULT_PROJECT_KEY,
}


def update(original, updates):
    # Update the default config with local config
    for key, value in updates.items():
        if isinstance(original.get(key), list) and isinstance(value, list):
            original[key] = value
        elif isinstance(original.get(key), dict) and isinstance(value, dict):
            original[key].update(value)
        else:
            original[key] = value
    return original


def load_config():
    local_config_path = Path(__file__).parent / "local_config.json"
    config = DEFAULT_CONFIG.copy()
    # Override with local configuration if it exists
    if local_config_path.exists():
        with open(local_config_path, "r", encoding=locale.getpreferredencoding(False)) as local_config_file:
            local_config = json.load(local_config_file)
            # Update the default config with local config
            update(config, local_config)
            # config.update(local_config)
    else:
        print(
            "No local configuration found. Default configuration will be used. Create a local_config.json file to override it."
        )
    return config


CONFIG = load_config()
os.environ["DKU_CURRENT_PROJECT_KEY"] = CONFIG.get("default_project_key", "")
os.environ["DKU_CUSTOM_WEBAPP_CONFIG"] = json.dumps(CONFIG.get("webapp_config"), ensure_ascii=False)


def get_setup_for_dataiku_client():
    return {
        "webapp_config": CONFIG.get("webapp_config"),
        "default_project_key": CONFIG.get("default_project_key"),
    }


dictConfig(
    {
        "version": 1,
        "formatters": {
            "default": {
                "format": "[%(asctime)s] %(levelname)s in %(module)s: %(message)s",
            }
        },
        "handlers": {
            "wsgi": {
                "class": "logging.StreamHandler",
                "stream": "ext://flask.logging.wsgi_errors_stream",
                "formatter": "default",
            }
        },
        "root": {"level": "INFO", "handlers": ["wsgi"]},
    }
)


def setup_dataiku_client():
    dataiku_setup = get_setup_for_dataiku_client()
    dataiku_api.setup(**dataiku_setup)
