from typing import Dict, Literal, Tuple, Union

import dataiku
from common.backend.models.base import UploadFileError, UploadFileResponse
from common.backend.utils.dataiku_api import dataiku_api
from common.backend.utils.file_extraction.upload import save_uploads
from common.backend.utils.file_extraction.upload_extraction import UploadFileExtractor
from common.backend.utils.picture_utils import b64encode_image_from_path
from common.llm_assist.log_decorators import log_http_request
from common.llm_assist.logging import logger
from flask import Blueprint, g, request

from ..utils import before_request, return_ko, return_ok

file_blueprint = Blueprint("file", __name__, url_prefix="/file")

webapp_config: Dict[str, str] = dataiku_api.webapp_config


@file_blueprint.before_request
def before_upload_request():
    before_request()


@file_blueprint.route("/summary_upload", methods=["POST"])
@log_http_request
def summary_upload_file() -> Tuple[str, Union[Literal[200], Literal[400]]]:
    """
    Upload file to a managed folder selected in webapp settings.
    Returns:
        Response: A Flask response object containing the upload data.
    """
    logger.info("summary upload file")
    auth_identifier: str = g.get("authIdentifier")

    list_of_extracted_summaries: UploadFileResponse
    try:
        if "files[]" not in request.files:
            raise Exception(UploadFileError.NO_SELECTED_FILE.value)
        files = request.files.getlist("files[]")
        language = request.form.get("language", "English")
        upload_extractor = UploadFileExtractor(files, auth_identifier, language)
        list_of_extracted_summaries = upload_extractor.extract()
        return return_ok(data=list_of_extracted_summaries), 200
    except Exception as e:
        logger.exception(f"Error occurred while uploading file: {e}")
        upload_extractor.clean_up()
        return return_ko(str(e)), 400


@file_blueprint.route("/upload", methods=["POST"])
@log_http_request
def upload_file() -> Tuple[str, Union[Literal[200], Literal[400]]]:
    logger.info("upload file")
    auth_identifier: str = g.get("authIdentifier")
    list_of_extracted_summaries: UploadFileResponse
    try:
        if "files[]" not in request.files:
            raise Exception(UploadFileError.NO_SELECTED_FILE.value)
        files = request.files.getlist("files[]")
        list_of_extracted_summaries = save_uploads(files, auth_identifier)
        return return_ok(data=list_of_extracted_summaries), 200
    except Exception as e:
        logger.exception(f"Error occurred while uploading file: {e}")
        # upload_extractor.clean_up()
        return return_ko(str(e)), 400



@file_blueprint.route("/delete", methods=["POST"])
@log_http_request
def delete_file() -> Tuple[str, Union[Literal[200], Literal[400]]]:
    # Delete a specific file from the managed folder
    request_as_json = request.get_json()
    try:
        file_path = request_as_json["file_path"]
        metadata_path = request_as_json.get("metadata_path")
        config: Dict[str, str] = dataiku_api.webapp_config
        upload_folder = config["upload_folder"]
        managedFolder = dataiku.Folder(upload_folder)
        managedFolder.delete_path(file_path)
        if metadata_path is not None:
            managedFolder.delete_path(metadata_path)
        return return_ok(), 200
    except Exception as e:
        logger.exception(f"Error occurred wile deleting file: {e}")
        return return_ko("Error occurred wile deleting file"), 400


@file_blueprint.route("/load", methods=["POST"])
@log_http_request
def load_file() -> Tuple[str, Union[Literal[200], Literal[400]]]:
    request_as_json = request.get_json()
    file_path = request_as_json["file_path"]
    file_type = request_as_json["file_type"]
    file_format = file_path.split(".")[-1]
    try:
        if file_type == "image":
            file_data = f"data:image/{file_format};base64,{b64encode_image_from_path(file_path)}"
            return return_ok(data={"file_path": file_path, "file_data": file_data}), 200
        else:
            return return_ko("File type not supported"), 400
    except Exception as e:
        logger.exception(f"Error occurred wile retrieving file: {e}")
        return return_ko("Error occurred wile retrieving file"), 400