import { computed, nextTick, ref } from 'vue'
import { useI18n } from 'vue-i18n'

export function useFileDropZone(
  uploadFileTypes: string | undefined,
  maxUploadSizeMB: number,
  maxNUploadFiles: number,
  profileLanguage: string | undefined,
  updateStatus: (x: string, y: string) => void
) {
  const { t } = useI18n()
  const selectedFilesRef = ref<File[]>([])
  const fileUploadButton = ref<HTMLButtonElement | null>(null)
  const dragging = ref(false)

  const fileInputEnabled = computed(() => !!uploadFileTypes?.length)
  type nullCheck = null | undefined

  const handleDrop = (event: DragEvent) => {
    event.preventDefault()
    dragging.value = false
    const files = event.dataTransfer?.files
    validateAndUpdateFileList(files)
    focusUploadButton()
  }

  const handleFileSelect = (event: Event) => {
    const target = event.target as HTMLInputElement
    const files = target.files
    validateAndUpdateFileList(files)
    focusUploadButton()
  }

  const validateAndUpdateFileList = (files: FileList | nullCheck) => {
    if (!files?.length) {
      console.error('No files detected on drop.')
      return
    }

    let existingFileListContainNewlySelectedFiles = false
    let newFileList = Array.from(files)

    const previouslySelectedFilesMap = new Map(
      selectedFilesRef.value.map((file) => [file.name, file])
    )

    newFileList.forEach((file) => {
      if (previouslySelectedFilesMap.has(file.name)) {
        existingFileListContainNewlySelectedFiles = true
      } else {
        previouslySelectedFilesMap.set(file.name, file)
      }
    })
    newFileList = Array.from(previouslySelectedFilesMap.values())

    if (existingFileListContainNewlySelectedFiles) {
      updateStatus('duplicate_files', t('duplicate_files'))
    }

    if (newFileList.length > maxNUploadFiles) {
      updateStatus(
        'max_number_of_files_reached',
        t('max_number_of_files_reached', {
          maxFilesAllowed: maxNUploadFiles
        })
      )

      return
    }

    // Loop over files and validate each one
    for (const file of newFileList) {
      // Validate file type
      if (!uploadFileTypes?.includes(file.type) || !file.type) {
        updateStatus('invalid_file_type', t('invalid_file_type'))
        return
      }
      // Validate file size
      if (file.size > maxUploadSizeMB * 1048576) {
        updateStatus(
          'file_too_large',
          t('file_too_large', { size: maxUploadSizeMB })
        )

        return
      }
    }
    selectedFilesRef.value = newFileList
  }

  const focusUploadButton = () => {
    nextTick(() => {
      fileUploadButton.value?.focus()
    })
  }

  const uploadFiles = async (files: File[]) => {
    const formData = new FormData()
    files.forEach((file) => {
      formData.append('files[]', file)
    })
    formData.append('language', profileLanguage ?? 'English')
    return formData
  }

  function handleDragEnter(event: DragEvent) {
    event.preventDefault()
    dragging.value = true
  }

  function handleDragLeave() {
    dragging.value = false
  }

  function handleDragOver(event: DragEvent) {
    event.preventDefault()
    dragging.value = false
  }

  return {
    selectedFilesRef,
    fileInputEnabled,
    handleDrop,
    handleFileSelect,
    uploadFiles,
    handleDragEnter,
    handleDragLeave,
    handleDragOver,
    dragging
  }
}
