from typing import Optional

from common.llm_assist.logging import logger
from common.solutions.chains.generic_summary_chain import GenericSummaryChain
from dataikuapi.dss.llm import DSSLLMCompletionQuery


class ImageSummaryChain(GenericSummaryChain):
    def __init__(self, img_b64: str, original_file_name: str, language: Optional[str]):
        self._img_b64 = img_b64
        self._original_file_name = original_file_name
        self._language = language

        self._completion = None


    @property
    def img_b64(self) -> str:
        return self._img_b64

    @property
    def original_file_name(self) -> str:
        return self._original_file_name

    @property
    def language(self) -> Optional[str]:
        return self._language


    def get_summary_completion(self) -> DSSLLMCompletionQuery:
        image_summary_system_prompt = """# Role and Guidelines
        Your role is to look at and summarize the image provided to you. This image has been uploaded to a webapp.
        Your summary should mention the contents of each section of the document if possible.
        You should provide your answer in JSON format with the following keys
        - "summary": str: summary of the whole image provided to you
        - "topics": Array[str]: array of all the main topics that appear in the image provided.
        - "questions": Array[str]: array of 3 questions which can be asked based on the image provided to you.
        The JSON object should have double quotes for keys and values. It is important to follow the JSON format strictly.
        Provide a JSON object only. Do not add any other explanations, decorations, or additional information beyond the JSON object.
        """
        image_summary_user_prompt = """Summarize the following extracted image
        Your JSON object:
        """
        if self.language:
            image_summary_system_prompt += f"""
        Your response should be in {self.language}
        """
        logger.debug(f"""
            Image Summary system Prompt: {image_summary_system_prompt}
            Image Summary user Prompt: {image_summary_user_prompt}
        """, log_conv_id=True)
        completion: DSSLLMCompletionQuery = self.init_completion()
        completion.with_message(image_summary_system_prompt, role="system")
        completion.with_message(image_summary_user_prompt, role="user")
        msg = completion.new_multipart_message()
        msg.with_text(f"Image file name: {self.original_file_name}").with_inline_image(self.img_b64).add()
        return completion
