from typing import Optional

from backend.database.base import db
from backend.database.models import AdminSettings


def get_admin_settings_for_user(user: str) -> Optional[AdminSettings]:
    """Get the admin settings record for a specific user (should be only one per user)."""
    return db.session.query(AdminSettings).filter(AdminSettings.user == user).first()


def upsert_admin_settings(user: str, settings: dict) -> AdminSettings:
    """
    Create or update admin settings for a user.
    There should only be one settings record per user.
    """
    existing = get_admin_settings_for_user(user)
    
    if existing:
        # Update existing settings
        existing.settings = settings
        db.session.commit()
        db.session.refresh(existing)
        return existing
    else:
        # Create new settings
        new_setting = AdminSettings(user=user, settings=settings)
        db.session.add(new_setting)
        db.session.commit()
        db.session.refresh(new_setting)
        return new_setting


def update_admin_setting_key(user: str, key: str, value: dict) -> AdminSettings:
    """
    Update a specific key in the admin settings JSON for a user.
    Creates the settings record if it doesn't exist.
    """
    existing = get_admin_settings_for_user(user)
    
    if existing:
        # Update specific key in existing settings
        current_settings = dict(existing.settings or {})
        current_settings[key] = value
        existing.settings = current_settings
        db.session.commit()
        db.session.refresh(existing)
        return existing
    else:
        # Create new settings with the key
        new_settings = {key: value}
        return upsert_admin_settings(user, new_settings)


def delete_admin_setting_key(user: str, key: str) -> Optional[AdminSettings]:
    """
    Delete a specific key from the admin settings JSON for a user.
    Returns the updated settings or None if user has no settings.
    """
    existing = get_admin_settings_for_user(user)
    
    if existing and existing.settings:
        current_settings = dict(existing.settings)
        if key in current_settings:
            del current_settings[key]
            existing.settings = current_settings
            db.session.commit()
            db.session.refresh(existing)
        return existing
    
    return None


def get_admin_setting_by_id(setting_id: int) -> Optional[AdminSettings]:
    """Get a specific admin setting by ID."""
    return db.session.query(AdminSettings).filter(AdminSettings.id == setting_id).first()


def get_all_admin_settings() -> list[AdminSettings]:
    """Get all admin settings."""
    return db.session.query(AdminSettings).all()


def delete_admin_setting(setting_id: int) -> bool:
    """Delete an admin setting by ID."""
    setting = get_admin_setting_by_id(setting_id)
    if setting:
        db.session.delete(setting)
        db.session.commit()
        return True
    return False
