from typing import List, Optional

from backend import schemas
from backend.database.base import db
from backend.database.models import Agent
from backend.utils.logging_utils import get_logger

logger = get_logger(__name__)


def _get_db_agent(agent_id: str) -> Agent | None:
    return db.session.query(Agent).filter(Agent.id == agent_id).first()

def _get_full_agent_details(db_agent: Agent) -> Optional[schemas.AgentRead]:
    from backend.constants import DRAFT_ZONE, PUBLISHED_ZONE
    from backend.utils.project_utils import get_agent_details, get_ua_project
    if not db_agent:
        return None
    agent_obj = None
    try:
        agent_obj = schemas.AgentRead.model_validate(db_agent)
        project = get_ua_project(db_agent.id)
        draft_agent_details = get_agent_details(project, DRAFT_ZONE)
        agent_obj.tools = draft_agent_details["tools"]
        agent_obj.kb_description = draft_agent_details["kb_description"]
        agent_obj.system_prompt = draft_agent_details["system_prompt"]
        agent_obj.llmid = draft_agent_details["llm_id"]
        agent_obj.name = draft_agent_details["name"]
        logger.debug(f"Agent[{agent_obj.name}] details: {draft_agent_details}")
        # Transform indexing field
        pub_details = get_agent_details(project, PUBLISHED_ZONE)
        # Transform published_version field - ensure it's an obj or None
        published_version = agent_obj.published_version
        if published_version:
            published_version.tools = pub_details["tools"]
            published_version.kb_description = pub_details["kb_description"]
            published_version.system_prompt = pub_details["system_prompt"]
            published_version.llmid = pub_details["llm_id"]
            published_version.name = pub_details["name"]
            agent_obj.published_version = published_version
            logger.debug(f"Published version details: {pub_details}")
        return agent_obj
    except Exception as e:
        logger.exception(f"Error getting full agent details for agent_id={db_agent.id}: {e}")
        return agent_obj

def get_agent(agent_id: str) -> Optional[schemas.AgentRead]:
    agent = _get_db_agent(agent_id)
    if not agent:
        return None
    return _get_full_agent_details(agent)




def get_all_agents() -> List[schemas.AgentRead]:
    agents = db.session.query(Agent).all()
    return [_get_full_agent_details(agent) for agent in agents]


def get_agents_by_owner(user_id: str) -> List[schemas.AgentRead]:
    agents = db.session.query(Agent).filter(Agent.owner == user_id).all()
    return [_get_full_agent_details(agent) for agent in agents]


def create_agent(agent: schemas.AgentCreate) -> schemas.AgentRead:
    new_agent = Agent(
        id=agent.id,
        owner=agent.owner,
        description=agent.description,
        sample_questions=agent.sample_questions,
        documents=agent.documents,
        indexing=agent.indexing,
        published_version=agent.published_version,
        published_at=agent.published_at,
        publishing_status=agent.publishing_status,
        publishing_job_id=agent.publishing_job_id,
    )
    # if get_agent(agent.id):
    #     raise ValueError(f"Agent '{agent.id}' already exists")
    db.session.add(new_agent)
    db.session.commit()
    db.session.refresh(new_agent)
    # TODO handle success and failure
    return schemas.AgentRead.model_validate(new_agent)


def update_agent(agent_id: str, agent_update: schemas.AgentUpdate) -> Optional[schemas.AgentRead]:
    db_agent = _get_db_agent(agent_id)
    if not db_agent:
        return None
    update_data = agent_update.model_dump(exclude_unset=True)
    for key, value in update_data.items():
        setattr(db_agent, key, value)
    db.session.commit()
    db.session.refresh(db_agent)
    return schemas.AgentRead.model_validate(db_agent)


def delete_agent(agent_id: str):
    db_agent = _get_db_agent(agent_id)
    if db_agent:
        db.session.delete(db_agent)
        db.session.commit()
        return True
    return False


def list_user_agent_owners() -> list[dict]:
    """
    Return a list of agent owners and their agent counts.
    """
    results = db.session.query(Agent.owner, db.func.count(Agent.id)).group_by(Agent.owner).order_by(Agent.owner).all()
    out = []
    for owner, count in results:
        display_name = ""
        email = ""
        try:
            from backend.utils.user_utils import get_user_info

            ui = get_user_info(owner)
            display_name = ui.display_name or ""
            email = ui.email or ""
        except Exception:
            pass
        out.append({"id": owner, "displayName": display_name, "email": email, "agentCount": int(count)})
    return out
