from sqlalchemy.orm.attributes import flag_modified

from backend import schemas
from backend.database.base import db
from backend.database.models import DraftConversation


def _get_db_draft_conv(agent_id: str, user_id: str):
    draft = db.session.query(DraftConversation).filter(
        DraftConversation.agent_id == agent_id,
        DraftConversation.user_id == user_id
    ).first()
    return draft

def get_draft_conversation(agent_id: str, user_id: str) -> schemas.DraftConversationRead | None:
    draft = _get_db_draft_conv(agent_id=agent_id, user_id=user_id)
    if draft:
        return schemas.DraftConversationRead.model_validate(draft)
    return None

def create_draft_conversation(draft_conversation: schemas.DraftConversationCreate) -> schemas.DraftConversationRead:
    new_draft_conversation = DraftConversation(**draft_conversation.model_dump())
    db.session.add(new_draft_conversation)
    db.session.commit()
    db.session.refresh(new_draft_conversation)
    return schemas.DraftConversationRead.model_validate(new_draft_conversation)

def upsert_draft_conversation(agent_id: str, user_id: str, convo: dict) -> None:
    """
    Insert or update a draft conversation for (agent_id, user_id).
    Sets created_at and last_modified timestamps as in store.py.
    """
    draft = _get_db_draft_conv(agent_id=agent_id, user_id=user_id)
    if draft:
        draft.convo = dict(convo)
        flag_modified(draft, "convo")
    else:
        draft = DraftConversation(
            agent_id=agent_id,
            user_id=user_id,
            convo=convo,
        )
        db.session.add(draft)
    db.session.commit()


def delete_draft_conversation(agent_id: str, user_id: str) -> None:
    """
    Delete the draft conversation for the given agent_id and user_id.
    """
    db.session.query(DraftConversation).filter(
        DraftConversation.agent_id == agent_id,
        DraftConversation.user_id == user_id
    ).delete()
    db.session.commit()