# backend/setup.py
import logging
import os
import warnings

import pydantic.warnings
from flask import Flask, jsonify
from pydantic import ValidationError

from backend.cache import cache
from backend.database.user_store import UserStore
from backend.fetch_api import fetch_api
from backend.routes import api
from backend.services.admin_config_service import setup_admin_config
from backend.socket import socketio
from backend.utils.logging_utils import configure_logging, get_logger
from backend.utils.ws_utils import before_request as auth_before_request
from backend.utils.ws_utils import setup_socketio

logger = get_logger(__name__)
os.environ["LANGCHAIN_TRACING_V2"] = "false"

import langsmith


def my_warning_handler(message, category, filename, lineno, file=None, line=None):
    if issubclass(category, pydantic.warnings.PydanticDeprecatedSince20):
        return  # ignore this warning
    if issubclass(category, langsmith.utils.LangSmithMissingAPIKeyWarning):
        return  # ignore this warning
    import sys

    sys.stderr.write(warnings.formatwarning(message, category, filename, lineno, line))


warnings.showwarning = my_warning_handler


class NoParsingFilter(logging.Filter):
    def filter(self, record):
        message = record.getMessage()
        return not message.endswith("AssertionError: write() before start_response")


def setup_app(app: Flask, db_url: str, workload_folder_path: str, dry=False) -> None:
    cache.init_app(app)
    app.register_blueprint(blueprint=fetch_api)
    app.register_blueprint(blueprint=api)
    app.before_request(auth_before_request)
    print(f"Setting up app with workload_folder_path: {workload_folder_path}")

    # Register Pydantic v2 validation error handler
    @app.errorhandler(ValidationError)
    def handle_pydantic_validation_error(e: ValidationError):
        errors = [
            {
                "field": ".".join(str(loc) for loc in err["loc"]),
                "message": err["msg"],
                "type": err["type"],
                "input": err.get("input"),
            }
            for err in e.errors()
        ]
        logger.error(f"Pydantic validation error: {errors}")
        return jsonify({"error": "Validation Error", "details": "Error occurred when validating data"}), 500

    # db_folder_path = get_workload_folder_path()
    if not (db_url):
        raise ValueError("DB URL Path is required to store SQLite.")
    logger.info(f"DB URL path: {db_url}")

    from backend.database import db, migrate

    app.config["SQLALCHEMY_DATABASE_URI"] = db_url
    app.config["SQLALCHEMY_TRACK_MODIFICATIONS"] = False

    # Check if using Snowflake OAuth (token may expire)
    is_snowflake_oauth = "snowflake://" in db_url and "authenticator=oauth" in db_url

    if is_snowflake_oauth:
        # For Snowflake OAuth: use custom creator to get fresh token on each new connection

        from backend.app_paths import get_db_url

        def get_fresh_snowflake_connection():
            """Create a new Snowflake connection with fresh OAuth token."""
            from sqlalchemy import create_engine

            fresh_url = get_db_url()  # Gets fresh token via get_oauth2_credential()
            temp_engine = create_engine(fresh_url)
            return temp_engine.raw_connection()

        app.config["SQLALCHEMY_ENGINE_OPTIONS"] = {
            "creator": get_fresh_snowflake_connection,
            "pool_pre_ping": True,  # Detect stale connections
            "pool_recycle": 1800,  # Force recycle every 30 min
        }
        logger.info("Snowflake OAuth: configured with fresh token creator")
    else:
        # For other databases: standard pooling
        app.config["SQLALCHEMY_ENGINE_OPTIONS"] = {
            "pool_pre_ping": True,
        }

    app.config["STORE"] = UserStore("", [])
    db.init_app(app)
    migrate.init_app(app, db)
    configure_logging()
    logging.getLogger("werkzeug").addFilter(NoParsingFilter())
    socketio.init_app(app)
    setup_socketio(socketio)
    if not dry:
        from backend.indexing_monitor import start_monitor, wake_monitor

        with app.app_context():
            setup_admin_config(workload_folder_path)
        start_monitor(app, interval=5)
        wake_monitor()

        from backend.services.publishing_service import PublishingService

        publishing_service = PublishingService(app=app)
        app.config["PUBLISHING_SERVICE"] = publishing_service
        publishing_service.resume_publishing_jobs()

