import { ref, watch, computed, type Ref } from 'vue'
import type { ConversationState } from '@/store/conversationStore'

interface BlockedAgent {
  agentId: string
  agentName: string
  content_violations?: string[]
  extraction_failures?: string[]
}

interface GuardrailsViolationState {
  /** Whether the guardrails violation modal should be shown */
  showModal: Ref<boolean>
  /** List of document names with content policy violations */
  contentViolations: Ref<string[]>
  /** List of document names with extraction failures */
  extractionFailures: Ref<string[]>
  /** List of blocked agents with their violations (for multi-agent scenarios) */
  blockedAgents: Ref<BlockedAgent[]>
  /** Reset the guardrails state (call when switching conversations) */
  reset: () => void
  /** Initialize known blocked attachments from existing conversation (call after loading) */
  initializeFromConversation: (conversation: ConversationState | undefined) => void
}

/**
 * Composable to manage guardrails violation state and detection.
 * Watches live events from a conversation and detects guardrails violations,
 * updating state and triggering the modal when needed.
 */
export function useGuardrailsViolation(
  activeConversation: Ref<ConversationState | null | undefined>
): GuardrailsViolationState {
  // Modal visibility
  const showModal = ref(false)

  // Violation data
  const contentViolations = ref<string[]>([])
  const extractionFailures = ref<string[]>([])
  const blockedAgents = ref<BlockedAgent[]>([])

  // Internal tracking
  const processedEventsCount = ref(0)
  const knownBlockedAttachmentNames = ref(new Set<string>())

  // Computed live events from the active conversation
  const liveEvents = computed(() => activeConversation.value?.liveEvents || [])

  // Watch for guardrails events in live events
  watch(
    liveEvents,
    (events) => {
      // Reset counter if the stream clears (new message started)
      if (events.length === 0) {
        processedEventsCount.value = 0
        return
      }

      // Filter to find all guardrails-related error events
      const guardrailsEvents = events.filter(
        (event) =>
          event.kind === 'guardrails_violation' || event.kind === 'guardrails_extraction_failed'
      )

      if (guardrailsEvents.length === 0) return

      // Only process if we have received NEW guardrails events since the last check
      if (guardrailsEvents.length <= processedEventsCount.value) {
        return
      }

      // Find the specific events (there might be one of each)
      const guardrailsViolationEvent = guardrailsEvents.find(
        (event) => event.kind === 'guardrails_violation'
      )
      const extractionFailureEvent = guardrailsEvents.find(
        (event) => event.kind === 'guardrails_extraction_failed'
      )

      if (!guardrailsViolationEvent && !extractionFailureEvent) {
        return
      }

      // Check if this is a blocked agents scenario (runtime guardrails)
      const blockedAgentsData =
        guardrailsViolationEvent?.data?.blocked_agents ||
        extractionFailureEvent?.data?.blocked_agents

      // Check if attachments have new violations
      const allViolations = [
        ...(guardrailsViolationEvent?.data?.failed_documents ?? []),
        ...(extractionFailureEvent?.data?.failed_documents ?? []),
      ]

      let hasNewViolations = false

      for (const name of allViolations) {
        if (!knownBlockedAttachmentNames.value.has(name)) {
          hasNewViolations = true
          knownBlockedAttachmentNames.value.add(name)
        }
      }

      const hasBlockedAgents = blockedAgentsData && blockedAgentsData.length > 0

      if (hasBlockedAgents) {
        // Multi-agent or runtime agent blocking scenario
        blockedAgents.value = blockedAgentsData
        // Clear the simple document arrays since we're using blocked_agents
        contentViolations.value = []
        extractionFailures.value = []
      } else {
        // Simple document list scenario (single agent with failed_documents)
        // We re-evaluate both arrays so that if an extraction failure arrives AFTER a content violation, both are shown
        contentViolations.value = guardrailsViolationEvent?.data?.failed_documents || []
        extractionFailures.value = extractionFailureEvent?.data?.failed_documents || []
        blockedAgents.value = []
      }

      // Display modal only if attachments guardrailsStatus changed or an agent was blocked
      if (hasBlockedAgents || (!hasBlockedAgents && hasNewViolations)) {
        showModal.value = true
      }
      // Update the counter so we don't re-process unless a 3rd event comes in
      processedEventsCount.value = guardrailsEvents.length
    },
    { deep: true }
  )

  /**
   * Reset the guardrails state. Call this when switching conversations.
   */
  function reset() {
    knownBlockedAttachmentNames.value.clear()
    processedEventsCount.value = 0
    contentViolations.value = []
    extractionFailures.value = []
    blockedAgents.value = []
    showModal.value = false
  }

  /**
   * Initialize known blocked attachments from existing conversation attachments.
   * Call this after loading a conversation to prevent re-triggering the modal
   * for already-known violations.
   */
  function initializeFromConversation(conversation: ConversationState | undefined) {
    if (!conversation) return

    const currentBlockedAttachments = (conversation.conversation_attachments || []).filter(
      (att) =>
        att.guardrailsStatus &&
        ['content_violation', 'extraction_failure'].includes(att.guardrailsStatus)
    )

    currentBlockedAttachments.forEach((att) =>
      knownBlockedAttachmentNames.value.add(att.document_name)
    )
  }

  return {
    showModal,
    contentViolations,
    extractionFailures,
    blockedAgents,
    reset,
    initializeFromConversation,
  }
}
