import { computed } from 'vue'
import { ThemeService } from '@/services/themeService'
import { useAdminConfigStore } from '@/store/adminConfigStore'
import { getBackendUrl } from '@/lib/backendUrl'
import defaultLogo from '@/assets/dataiku-logo.png'

/**
 * Composable for reactive access to theme customization settings.
 * Assets are read from ThemeService to ensure unsaved changes don't leak outside admin page.
 */
export function useTheme() {
  const adminStore = useAdminConfigStore()

  const buildAdminAssetUrl = (path: string | null | undefined, managedFolderId?: string) => {
    if (!path) return null
    const params = new URLSearchParams({ path })
    if (managedFolderId) {
      params.set('managedFolderId', managedFolderId)
    }
    const baseUrl = getBackendUrl().replace(/\/$/, '')
    return `${baseUrl}/api/downloads/admin/assets?${params.toString()}`
  }

  const logoUrl = computed(() => {
    const path = ThemeService.getLeftPanelLogoPathRef().value
    return buildAdminAssetUrl(path, adminStore.draft?.adminUploadsManagedFolder) || defaultLogo
  })

  const homepageImageUrl = computed(() => {
    const path = ThemeService.getHomepageImagePathRef().value
    return buildAdminAssetUrl(path, adminStore.draft?.adminUploadsManagedFolder) || defaultLogo
  })

  const logoSrc = computed(() => logoUrl.value || defaultLogo)
  const homepageImageSrc = computed(() => homepageImageUrl.value || defaultLogo)

  const homepageTitle = computed(() => {
    return ThemeService.getHomepageTitleRef().value || 'Welcome to Agent Hub'
  })

  const themeColor = computed(() => {
    return adminStore.draft?.themeColor || null
  })

  const themeColorMode = computed(() => {
    return adminStore.draft?.themeColorMode || 'automatic'
  })

  const customCssPath = computed(() => {
    return adminStore.draft?.customCssPath || null
  })

  return {
    logoUrl,
    logoSrc,
    homepageImageUrl,
    homepageImageSrc,
    homepageTitle,
    themeColor,
    themeColorMode,
    customCssPath,
  }
}
