import snowflakeIcon from '@/assets/special-icon/snowflake_logo.svg'
import bedrockIcon from '@/assets/special-icon/bedrock_logo.svg'
import databricksIcon from '@/assets/special-icon/databricks_logo.svg'

/**
 * Plugin Agent Type Constants
 * Maps plugin agent types to their corresponding icon paths
 */
export const PLUGIN_AGENT_TYPES = {
  SNOWFLAKE: 'agent_snowflake-tools_cortex-agent',
  BEDROCK: 'agent_aws-bedrock-agents_bedrock-agent',
  DATABRICKS: 'agent_databricks-tools_databricks-agents',
} as const

export const PLUGIN_AGENT_ICONS: Record<string, string> = {
  [PLUGIN_AGENT_TYPES.SNOWFLAKE]: snowflakeIcon,
  [PLUGIN_AGENT_TYPES.BEDROCK]: bedrockIcon,
  [PLUGIN_AGENT_TYPES.DATABRICKS]: databricksIcon,
}

/**
 * Get the icon path for a given plugin agent type
 * @param pluginAgentType - The plugin agent type identifier
 * @returns The icon path if found, undefined otherwise
 */
export function getPluginAgentIcon(pluginAgentType?: string): string | undefined {
  if (!pluginAgentType) return undefined
  return PLUGIN_AGENT_ICONS[pluginAgentType]
}

/**
 * Get the display name for a given plugin agent type
 * @param pluginAgentType - The plugin agent type identifier
 * @returns The display name (e.g., "Snowflake", "Bedrock") if found, undefined otherwise
 */
export function getPluginAgentTypeName(pluginAgentType?: string): string | undefined {
  if (!pluginAgentType) return undefined
  if (pluginAgentType === PLUGIN_AGENT_TYPES.SNOWFLAKE) {
    return 'Snowflake'
  }
  if (pluginAgentType === PLUGIN_AGENT_TYPES.BEDROCK) {
    return 'Bedrock'
  }
  if (pluginAgentType === PLUGIN_AGENT_TYPES.DATABRICKS) {
    return 'Databricks'
  }
  return undefined
}

/**
 * Get the icon class for a given plugin agent type
 * @param pluginAgentType - The plugin agent type identifier
 * @returns The icon class if found, undefined otherwise
 */
export function getPluginAgentIconClass(pluginAgentType?: string): string | undefined {
  if (!pluginAgentType) return undefined
  if (pluginAgentType === PLUGIN_AGENT_TYPES.SNOWFLAKE) {
    return 'dku-icon-snowflake-brand-16'
  }
  if (pluginAgentType === PLUGIN_AGENT_TYPES.BEDROCK) {
    return 'dku-icon-amazon-aws-bedrock-16'
  }
  if (pluginAgentType === PLUGIN_AGENT_TYPES.DATABRICKS) {
    return 'dku-icon-databricks-16'
  }
  return undefined
}
