import type { EventLogEntry } from '@/types/messages'

export function filterAndCopyEvents(existingEvents: EventLogEntry[], newEvents: EventLogEntry[]) {
  if (!newEvents) return

  for (const event of newEvents) {
    const lastFiltered =
      existingEvents.length > 0 ? existingEvents[existingEvents.length - 1] : null

    if (event.eventKind === 'agent_reasoning') {
      if (lastFiltered?.eventKind === 'AGENT_THINKING') {
        existingEvents.pop()
      }
      existingEvents.push(event)
    } else if (event.eventKind === 'AGENT_THINKING') {
      if (
        lastFiltered?.eventKind !== 'AGENT_THINKING' &&
        lastFiltered?.eventKind !== 'agent_reasoning'
      ) {
        existingEvents.push(event)
      }
    } else {
      if (
        lastFiltered &&
        lastFiltered.eventKind === event.eventKind &&
        JSON.stringify(lastFiltered.eventData) === JSON.stringify(event.eventData)
      ) {
        // prevent adding consecutive duplicate events (e.g.: "using tool xx" spam)
        continue
      }
      existingEvents.push(event)
    }
  }
}

export function processReasoningLiveEvents(
  liveEvents: Array<{ kind: string; data: any }>,
  currentReasoningAgentName: string | null,
  reasoningText: string,
  liveReasoningAgents: Record<string, boolean>
): {
  reasoningBuffer: string
  reasoningAgentName: string | null
  liveReasoningAgents: Record<string, boolean>
} {
  /*
        Processes events in sequence in order to rebuild and display
        the complete reasoning session of the latest agent that started one.
    */
  for (const event of liveEvents) {
    const eventAgentName = event.data?.agentName
    /* 
            liveReasoningAgents is the record holding the status
            of known agents which are currently reasoning. As they can reason
            at  the same time, doing so allow us to know if a reasoning event is part of 
            an already started reasoning session (and display it or ignore it) or if it's 
            a new reasoning session. 
        */

    if (event.kind === 'agent_reasoning') {
      if (liveReasoningAgents[eventAgentName] !== true) {
        // if an agent start a reasoning session we reset the buffer
        // and start to display it. Doing so we display the last reasoning agent.
        reasoningText = ''
        currentReasoningAgentName = eventAgentName
        liveReasoningAgents[eventAgentName] = true
      }

      if (eventAgentName === currentReasoningAgentName) {
        // it's the agent we're displaying so we aggregate the reasoning text
        const evtAggregatedText = ((event.data?.parts as any[]) ?? [])
          .filter((part) => part.type === 'TEXT')
          .map((part) => part.text)
          .join('')
        reasoningText += evtAggregatedText
      }
    } else {
      if (eventAgentName && liveReasoningAgents[eventAgentName]) {
        // if an agent which was reasoning makes something else it means
        // it's not reasoning anymore
        liveReasoningAgents[eventAgentName] = false
        if (eventAgentName === currentReasoningAgentName) {
          reasoningText = ''
          currentReasoningAgentName = null
        }
      }
    }
  }

  return {
    reasoningBuffer: reasoningText,
    reasoningAgentName: currentReasoningAgentName,
    liveReasoningAgents,
  }
}
