import type { ToolValidationRequest } from '@/types/messages'

/**
 * Extracts and deduplicates tool validation requests from live-streaming events.
 * @param liveEvents - The array of live events from the conversation stream.
 * @returns An array of unique ToolValidationRequest objects.
 */
export function extractLiveToolValidationRequests(liveEvents: any[]): ToolValidationRequest[] {
  if (!liveEvents || liveEvents.length === 0) {
    return []
  }

  const validationEvents = liveEvents.filter(
    (event) => event.kind === 'toolValidationRequests' && event.data
  )

  // Use a Map to automatically handle deduplication by request ID
  const requestsMap = new Map<string, ToolValidationRequest>()
  for (const event of validationEvents) {
    for (const req of event.data.requests) {
      // We should be defensive and check the shape of event.data
      if (req && typeof req.id !== 'undefined' && req.toolCall) {
        requestsMap.set(String(req.id), req as ToolValidationRequest)
      }
    }
  }

  return Array.from(requestsMap.values())
}
