import { ref } from 'vue'
import type { AdminConfig } from '@/types/admin-config'
import { getBackendUrl } from '@/lib/backendUrl'

/**
 * Service for applying theme customizations (colors, CSS, assets).
 * Components should read theme assets from this service to ensure proper isolation.
 */
export class ThemeService {
  private static customCssLinkId = 'custom-theme-css'

  private static _leftPanelLogoPath = ref<string | undefined>(undefined)
  private static _homepageImagePath = ref<string | undefined>(undefined)
  private static _homepageTitle = ref<string | undefined>(undefined)

  static applyTheme(config: AdminConfig | undefined): void {
    if (!config) return

    const root = document.documentElement

    if (config.themeColor) {
      root.style.setProperty('--primary', config.themeColor)
    } else {
      root.style.removeProperty('--primary')
    }

    if (config.customCssPath) {
      const params = new URLSearchParams({ path: config.customCssPath })
      if (config.adminUploadsManagedFolder) {
        params.set('managedFolderId', config.adminUploadsManagedFolder)
      }
      const baseUrl = getBackendUrl().replace(/\/$/, '')
      const cssUrl = `${baseUrl}/api/downloads/admin/assets?${params.toString()}`
      this.loadCustomCSS(cssUrl)
    } else {
      this.removeCustomCSS()
    }

    this._leftPanelLogoPath.value = config.leftPanelLogoPath
    this._homepageImagePath.value = config.homepageImagePath
    this._homepageTitle.value = config.homepageTitle

    if (config.customFontEnabled) {
      console.log('Custom fonts feature is not yet implemented')
    }
  }

  static getLeftPanelLogoPathRef() {
    return this._leftPanelLogoPath
  }

  static getHomepageImagePathRef() {
    return this._homepageImagePath
  }

  static getHomepageTitleRef() {
    return this._homepageTitle
  }

  static loadCustomCSS(path: string): void {
    this.removeCustomCSS()

    const link = document.createElement('link')
    link.id = this.customCssLinkId
    link.rel = 'stylesheet'
    link.href = path
    link.type = 'text/css'

    document.head.appendChild(link)
  }

  static removeCustomCSS(): void {
    const existingLink = document.getElementById(this.customCssLinkId)
    if (existingLink) {
      existingLink.remove()
    }
  }

  static resetTheme(): void {
    const root = document.documentElement
    root.style.removeProperty('--primary')
    this.removeCustomCSS()
  }

  static getCurrentThemeColor(): string | null {
    return document.documentElement.style.getPropertyValue('--primary') || null
  }
}
