import { defineStore } from 'pinia'
import { ref, computed } from 'vue'
import { API } from '@/Api'
import type { AdminConfig } from '@/types/admin-config'
import { usePrefStore } from '@/store/prefStore'
import { useToolsStore } from '@/store/toolsStore'
import { useAgentCatalogStore } from '@/store/agentCatalogStore'
import { usePromptStore } from '@/store/promptStore'
import { getSocketId } from '@/composables/useSocket'

type WithMeta<T> = T & { _meta?: { etag?: string } }

function deepClone<T>(v: T): T {
  return JSON.parse(JSON.stringify(v))
}

export const useAdminConfigStore = defineStore('adminConfig', () => {
  const original = ref<WithMeta<AdminConfig> | undefined>(undefined)
  const draft = ref<WithMeta<AdminConfig> | undefined>(undefined)

  const loaded = ref(false)
  const loading = ref(false)
  const savingConfig = ref(false)
  const savingLibrary = ref(false)
  const error = ref<string | null>(null)

  // Admin settings presence tracking
  const activeEditors = ref<Array<{ userId: string; joinedAt: string }>>([])

  const dirty = computed(
    () =>
      JSON.stringify(original.value?.promptsAttributesById ?? {}) !==
        JSON.stringify(draft.value?.promptsAttributesById ?? {}) ||
      JSON.stringify({
        ...(original.value ?? {}),
        promptsAttributesById: undefined,
        _meta: undefined,
      }) !==
        JSON.stringify({
          ...(draft.value ?? {}),
          promptsAttributesById: undefined,
          _meta: undefined,
        })
  )

  const sectionDirty = (section: string) => {
    if (!loaded.value || !original.value || !draft.value) return false
    const sectionKey = section === 'prompts' ? 'promptsAttributesById' : section
    return (
      JSON.stringify((original.value as any)[sectionKey]) !==
      JSON.stringify((draft.value as any)[sectionKey])
    )
  }

  const isSectionDirty = (sectionId: string): boolean => {
    switch (sectionId) {
      case 'system':
        return (
          sectionDirty('agentHubLLM') ||
          sectionDirty('agentHubOptionalInstructions') ||
          sectionDirty('orchestrationMode') ||
          sectionDirty('logsLevel') ||
          sectionDirty('permanentlyDeleteMessages')
        )
      case 'enterprise-agents':
        return sectionDirty('enterpriseAgents')
      case 'my-agents':
        return (
          sectionDirty('myAgentsEnabled') ||
          sectionDirty('myAgentsTextCompletionModels') ||
          sectionDirty('myAgentsEmbeddingModel') ||
          sectionDirty('myAgentsFsConnection') ||
          sectionDirty('myAgentsFolder') ||
          sectionDirty('myAgentsNumDocs') ||
          sectionDirty('myAgentsManagedTools') ||
          sectionDirty('myAgentsEnableGroupsRestriction') ||
          sectionDirty('myAgentsExcludedShareGroups')
        )
      case 'charts':
        return (
          sectionDirty('chartsGenerationMode') ||
          sectionDirty('chartsTextCompletionModel') ||
          sectionDirty('chartsMaxArtifactsSize')
        )
      case 'prompts':
        return (
          sectionDirty('myAgentsEnablePromptLibrary') ||
          sectionDirty('promptsAttributesById') ||
          sectionDirty('promptsAssetLibraryCollectionIds')
        )
      case 'conversation-files':
        return (
          sectionDirty('enableDocumentUpload') ||
          sectionDirty('extractionMode') ||
          sectionDirty('textExtractionType') ||
          sectionDirty('maxImagesInConversation') ||
          sectionDirty('uploadManagedFolder') ||
          sectionDirty('conversationVisionLLM')
        )
      case 'user-experience':
        return (
          sectionDirty('smartMode') ||
          sectionDirty('allowDisableAgents') ||
          sectionDirty('conversationStarterExamples')
        )
      case 'look-and-feel':
        return (
          sectionDirty('themeColor') ||
          sectionDirty('themeColorMode') ||
          sectionDirty('leftPanelLogoPath') ||
          sectionDirty('homepageImagePath') ||
          sectionDirty('homepageTitle') ||
          sectionDirty('customCssPath') ||
          sectionDirty('customFontEnabled') ||
          sectionDirty('adminUploadsManagedFolder')
        )
      default:
        return false
    }
  }

  const saving = computed(() => savingConfig.value || savingLibrary.value)

  async function load() {
    loading.value = true
    error.value = null
    try {
      const cfg = await API.getAdminConfig()
      original.value = cfg.data
      draft.value = deepClone(cfg.data)
      loaded.value = true
    } catch (e: any) {
      error.value = e?.message || 'Failed to load config'
      console.error('Error loading admin config / prompts library:', e)
    } finally {
      loading.value = false
    }
  }

  async function reloadAdminAffectedStores() {
    // Reload stores that depend on admin config settings
    const prefStore = usePrefStore()
    const toolsStore = useToolsStore()
    const agentCatalog = useAgentCatalogStore()
    const promptStore = usePromptStore()

    // Reset prompt store to force reload (since loadTemplates has a guard)
    promptStore.reset()

    await Promise.all([
      prefStore.reload(),
      toolsStore.refresh(),
      agentCatalog.load(true),
      promptStore.loadCollections(),
      promptStore.loadTemplates(),
    ])
  }

  async function save() {
    if (!draft.value) return false
    savingConfig.value = true
    error.value = null
    try {
      // Compute removed Look & Feel asset paths (for post-save cleanup)
      const removedLookAndFeelAssets: string[] = []
      if (original.value) {
        const originalCfg = original.value as AdminConfig
        const draftCfg = draft.value as AdminConfig
        const assetKeys: (keyof AdminConfig)[] = [
          'leftPanelLogoPath',
          'homepageImagePath',
          'customCssPath',
        ]
        for (const key of assetKeys) {
          const before = (originalCfg as any)[key] as string | undefined | null
          const after = (draftCfg as any)[key] as string | undefined | null
          if (before && before !== after) {
            removedLookAndFeelAssets.push(before)
          }
        }
      }

      const etag = original.value?._meta?.etag
      const socketId = getSocketId()
      await API.saveAdminConfig({
        config: draft.value as AdminConfig,
        etag,
        socketId: socketId || undefined,
      })
      const cfg = await API.getAdminConfig()
      original.value = cfg.data
      draft.value = deepClone(cfg.data)

      // After successful save, clean up any removed Look & Feel assets
      try {
        const managedFolderId = (draft.value as AdminConfig | undefined)?.adminUploadsManagedFolder
        if (managedFolderId && removedLookAndFeelAssets.length > 0) {
          await API.cleanupDraftAssets(removedLookAndFeelAssets, managedFolderId)
        }
      } catch (cleanupErr) {
        console.error('Error cleaning up removed Look & Feel assets:', cleanupErr)
      }

      await reloadAdminAffectedStores()

      return true
    } catch (e: any) {
      error.value = e?.message || 'Failed to save admin config'
      console.error('Error saving admin config:', e)
      return false
    } finally {
      savingConfig.value = false
    }
  }

  function isPromptEnabled(id: string): boolean {
    const attrs = draft.value?.promptsAttributesById ?? {}
    const override = (attrs as Record<string, { enabled?: boolean }>)[id]
    return override?.enabled !== false
  }

  function setPromptEnabled(id: string, enabled: boolean) {
    if (!draft.value) return
    const byId = (draft.value.promptsAttributesById ||= {})
    if (enabled) {
      delete (byId as Record<string, any>)[id]
    } else {
      ;(byId as Record<string, any>)[id] = { enabled: false }
    }
  }

  function discard() {
    if (original.value) {
      draft.value = deepClone(original.value)
    }
  }

  function reset() {
    original.value = undefined
    draft.value = undefined
    loaded.value = false
    loading.value = false
    savingLibrary.value = false
    savingConfig.value = false
    error.value = null
    activeEditors.value = []
  }

  function setActiveEditors(editors: Array<{ userId: string; joinedAt: string }>) {
    activeEditors.value = editors
  }

  return {
    original,
    draft,
    loaded,
    loading,
    savingConfig,
    savingLibrary,
    error,
    activeEditors,

    dirty,
    sectionDirty,
    isSectionDirty,
    saving,

    load,
    save,
    reloadAdminAffectedStores,
    discard,
    reset,

    isPromptEnabled,
    setPromptEnabled,
    setActiveEditors,
  }
})
