import { defineStore } from 'pinia'
import { ref } from 'vue'
import { API } from '@/Api'
import type { Prompt } from '@/types/admin-config'

export const usePromptStore = defineStore('prompt', () => {
  // State
  const promptTemplates = ref<Prompt[]>([])
  const loading = ref(false)
  const error = ref<string | null>(null)
  const isLoaded = ref(false)

  // Collections state
  const collections = ref<{ id: string; name: string }[]>([])
  const collectionsLoading = ref(false)
  const collectionsError = ref<string | null>(null)
  const isCollectionsLoaded = ref(false)

  // Actions
  async function loadTemplates() {
    // Don't reload if already loaded
    if (isLoaded.value) {
      return
    }

    loading.value = true
    error.value = null

    try {
      const response = await API.getPromptsLibraryForUsers()
      promptTemplates.value = response.data.prompts || []
      isLoaded.value = true
    } catch (err: any) {
      error.value = err.message || 'Failed to load prompt templates'
      console.error('Error loading prompt templates:', err)
      promptTemplates.value = []
    } finally {
      loading.value = false
    }
  }

  async function loadCollections() {
    if (isCollectionsLoaded.value) {
      return
    }

    collectionsLoading.value = true
    collectionsError.value = null

    try {
      collections.value = await API.getCollectionsForUsers()
      isCollectionsLoaded.value = true
    } catch (err: any) {
      collectionsError.value = err.message || 'Failed to fetch collections'
      console.error('Failed to fetch collections', err)
      collections.value = []
    } finally {
      collectionsLoading.value = false
    }
  }

  function searchTemplates(query: string): Prompt[] {
    if (!promptTemplates.value) return []

    if (!query.trim()) {
      return promptTemplates.value
    }

    const searchTerm = query.toLowerCase()
    return promptTemplates.value.filter(
      (template) =>
        template.name.toLowerCase().includes(searchTerm) ||
        template.description.toLowerCase().includes(searchTerm) ||
        template.content.toLowerCase().includes(searchTerm)
    )
  }

  function getTemplateById(id: string): Prompt | undefined {
    return promptTemplates.value?.find((template) => template.id === id)
  }

  // Reset state (useful for testing or when switching contexts)
  function reset() {
    // Reset Prompts
    promptTemplates.value = []
    loading.value = false
    error.value = null
    isLoaded.value = false

    // Reset Collections
    collections.value = []
    collectionsLoading.value = false
    collectionsError.value = null
    isCollectionsLoaded.value = false
  }

  return {
    // State
    promptTemplates,
    loading,
    error,
    isLoaded,
    collections,
    collectionsLoading,
    collectionsError,
    isCollectionsLoaded,

    // Actions
    loadTemplates,
    loadCollections,
    searchTemplates,
    getTemplateById,
    reset,
  }
})
