import os
import subprocess
import sys
import time
from contextlib import contextmanager

import dataiku
from backend.app_paths import get_db_schema, get_db_url, get_tables_prefix, get_workload_folder_path

schema = get_db_schema()
if schema:
    os.environ["DB_SCHEMA"] = schema

tables_prefix = get_tables_prefix()
if tables_prefix:
    os.environ["TABLES_PREFIX"] = tables_prefix

from backend.execution import Execution
from backend.setup import setup_app
from backend.utils.logging_utils import get_logger

logger = get_logger(__name__)

DB_URL = get_db_url()


@contextmanager
def timed_step(step_name: str):
    """Context manager to time and log a step.
    
    Usage:
        with timed_step("Database migration"):
            run_migration()
    """
    logger.info("[%s] Starting...", step_name)
    start = time.perf_counter()
    try:
        yield
        duration = time.perf_counter() - start
        logger.info("[%s] Completed in %.2fs", step_name, duration)
    except Exception as e:
        duration = time.perf_counter() - start
        logger.error("[%s] Failed after %.2fs: %s", step_name, duration, e)
        raise


def _log_app_tables(engine, schema):
    """Log all app tables from SQLAlchemy metadata after migration."""
    try:
        from backend.database.base import db
        from sqlalchemy import inspect

        inspector = inspect(engine)
        existing_tables = set(inspector.get_table_names(schema=schema))

        # Get expected tables from SQLAlchemy metadata (source of truth)
        # Import models to ensure metadata is populated
        from backend.database import models  # noqa: F401

        expected_tables = sorted(db.metadata.tables.keys())

        logger.info("Agent Hub tables (schema=%s):", schema)
        for table in expected_tables:
            status = "✓" if table in existing_tables else "✗"
            logger.info("  %s %s", status, table)
    except Exception as e:
        logger.warning("Could not list app tables: %s", e)


def _setup_alembic_env():
    env = {
        "DATABASE_URL": DB_URL,
        "DKU_CURRENT_PROJECT_KEY": dataiku.api_client().get_default_project().project_key,
        "DB_FOLDER_PATH": get_workload_folder_path(),
    }
    if schema:
        env["DB_SCHEMA"] = schema
    if tables_prefix:
        env["TABLES_PREFIX"] = tables_prefix
    # Set PYTHONPATH to include plugin python-lib and all sys.path entries
    plugin_python_lib = os.path.abspath(os.path.join(os.path.dirname(__file__), "../../python-lib"))
    logger.info("Plugin python-lib path: %s", plugin_python_lib)
    sys_path = ":".join(sys.path)
    # TODO SQLAlchemy/Pydantic: Check if this is correct in all cases
    env["PYTHONPATH"] = f"{plugin_python_lib}:{sys_path}"
    python_executable = sys.executable
    logger.info("Using executable: %s", python_executable)
    # Get plugin root from DKU_CUSTOM_RESOURCE_FOLDER (points to <plugin>/resource)
    # Go up one level to get plugin root where alembic.ini is located
    resource_folder = os.environ.get("DKU_CUSTOM_RESOURCE_FOLDER")
    if resource_folder:
        project_root = os.path.dirname(resource_folder)
    else:
        # Fallback for local development
        project_root = Execution("").exec_path
    logger.info("Project root for Alembic: %s", project_root)

    return python_executable, env, project_root


def run_alembic_downgrade():
    python_executable, env, project_root = _setup_alembic_env()
    logger.info("Running Alembic downgrade")

    try:
        logger.info("Downgrading the database to the base revision")
        # run downgrade; '-1' will downgrade to the previous revision
        subprocess.run([python_executable, "-m", "alembic", "downgrade", "-1"], check=True, env=env, cwd=project_root)
    except Exception as e:
        logger.exception(f"Failed to run Alembic downgrade: {e}")
        raise Exception("Alembic downgrade failed")


def run_alembic_upgrade():
    python_executable, env, project_root = _setup_alembic_env()
    try:
        from sqlalchemy import create_engine

        engine = create_engine(DB_URL)
        logger.info("Starting Alembic upgrade for DB: %s", DB_URL)
        
        result = subprocess.run(
            [python_executable, "-m", "alembic", "upgrade", "head"],
            check=True,
            env=env,
            cwd=project_root,
            capture_output=True,
            text=True,
        )
        
        # Always log migration output (both stdout and stderr contain useful info)
        if result.stdout:
            for line in result.stdout.strip().split('\n'):
                if line:
                    logger.info("%s", line)
        if result.stderr:
            for line in result.stderr.strip().split('\n'):
                if line:
                    logger.info("%s", line)
        
        # Check if any migrations were applied
        if "Running upgrade" in result.stdout or "Running upgrade" in result.stderr:
            logger.info("Migration applied successfully.")
        else:
            logger.info("Database already at latest revision.")

        _log_app_tables(engine, schema)
        return result
    except subprocess.CalledProcessError as e:
        logger.error("Alembic migration failed with exit code %s", e.returncode)
        logger.error(f"STDOUT: {e.stdout}")
        logger.error(f"STDERR: {e.stderr}")
        raise Exception(f"Alembic migration failed")
    except Exception as e:
        logger.exception(f"Failed to run Alembic migration: {e}")
        raise Exception("Alembic migration failed")


# Startup sequence with timing
logger.info("========== Backend Startup ==========")
_total_start = time.perf_counter()

with timed_step("Database migration"):
    run_alembic_upgrade()

with timed_step("App setup"):
    setup_app(app, DB_URL, get_workload_folder_path())

logger.info("========== Startup complete in %.2fs ==========", time.perf_counter() - _total_start)
