from dataiku.llm.python import BaseLLM
from agenticllmcouncil import (
    DEFAULT_CHAIRMAN_SYSTEM_PROMPT_TEMPLATE,
    DEFAULT_REVIEW_SYSTEM_PROMPT_TEMPLATE,
    get_members_first_opinions,
    get_members_reviews,
    get_chairman_final_answer,
    validate_config,
)


class CouncilLLMProvider(BaseLLM):
    def __init__(self):
        self.member_ids = []
        self.chairman_id = None
        self.member_review_prompt = DEFAULT_REVIEW_SYSTEM_PROMPT_TEMPLATE
        self.chairman_final_answer_prompt = DEFAULT_CHAIRMAN_SYSTEM_PROMPT_TEMPLATE

    def set_config(self, config: dict, plugin_config: dict) -> None:
        """Apply configuration settings."""
        validate_config(config)
        self.member_ids = [member.get(f"member") for member in config.get(f"members") if member.get(f"member")]
        self.chairman_id = config.get("chairman_id")

        if config.get("customize_system_prompt"):
            self.member_review_prompt = config.get("member_review_prompt")
            self.chairman_final_answer_prompt = config.get("chairman_final_answer_prompt")

    def process(self, query, settings, trace):
        """Execute the full council workflow synchronously."""
        full_text = ""
        for chunk_data in self.process_stream(query, settings, trace):
            if "chunk" in chunk_data:
                text_part = chunk_data["chunk"].get("text")
                if text_part:
                    full_text += text_part
        return {"text": full_text}

    def process_stream(self, query, settings, trace):
        """Execute the council workflow and stream the chairman's response."""
        messages = query.get("messages", [])

        first_opinions = get_members_first_opinions(messages, self.member_ids, trace)
        reviews = get_members_reviews(messages, first_opinions, self.member_ids, self.member_review_prompt, trace)
        final_answer_stream = get_chairman_final_answer(
            messages, first_opinions, reviews, self.chairman_id, self.chairman_final_answer_prompt
        )

        for item in final_answer_stream:
            if item.type in ["text", "content"]:
                yield {"chunk": {"text": item.text or ""}}

            elif item.type == "footer":
                if trace:
                    sub_trace = trace.subspan("CHAIRMAN_ANSWER_STEP")
                    sub_trace.append_trace(item.trace)
