import concurrent.futures
from .utils import _call_model, _prepare_messages
from .prompts import CHAIRMAN_USER_PROMPT_TEMPLATE, REVIEW_USER_PROMPT_TEMPLATE


def get_members_first_opinions(messages: list, member_ids: list, trace) -> dict:
    """Collect initial independent first opinions from all members."""
    first_opinions = {}
    first_opinions_traces = {}

    with concurrent.futures.ThreadPoolExecutor(max_workers=len(member_ids)) as executor:
        future_to_id = {executor.submit(_call_model, eid, messages): (i, eid) for i, eid in enumerate(member_ids)}

        for f in concurrent.futures.as_completed(future_to_id):
            index, eid = future_to_id[f]
            result = f.result()
            first_opinions[index] = {"id": eid, "text": result.text}
            first_opinions_traces[index] = result.trace

    if trace:
        sub_trace = trace.subspan("MEMBERS_FIRST_OPINION_STEP")
        for index, data in first_opinions_traces.items():
            sub_trace.append_trace(data)

    return first_opinions


def get_members_reviews(
    messages: list, members_first_opinions: dict, member_ids: list, member_review_prompt: str, trace
) -> dict:
    """Gather peer critiques on the initial first opinions."""
    reviews = {}
    review_traces = {}

    with concurrent.futures.ThreadPoolExecutor(max_workers=len(member_ids)) as executor:
        future_to_id = {}
        for i in range(len(member_ids)):
            reviewer_id = member_ids[i]

            first_opinions_text = ""
            for index, data in members_first_opinions.items():
                first_opinions_text += f"\n[Member {index+1}]\n{data['text']}\n"

            review_messages = _prepare_messages(
                messages,
                member_review_prompt,
                REVIEW_USER_PROMPT_TEMPLATE.format(first_opinions=first_opinions_text),
            )
            future_to_id[executor.submit(_call_model, reviewer_id, review_messages)] = (i, reviewer_id)

        for f in concurrent.futures.as_completed(future_to_id):
            index, reviewer_id = future_to_id[f]
            result = f.result()
            reviews[index] = {"id": reviewer_id, "text": result.text}
            review_traces[index] = result.trace

    if trace:
        sub_trace = trace.subspan("MEMBERS_REVIEW_STEP")
        for index, data in review_traces.items():
            sub_trace.append_trace(data)

    return reviews


def get_chairman_final_answer(
    messages: list, members_first_opinions: dict, reviews: dict, chairman_id: str, chairman_final_answer_prompt: str
):
    """Synthesize first opinions and reviews into a final answer using the Chairman."""
    first_opinions_text = ""
    for index, data in members_first_opinions.items():
        first_opinions_text += f"\n[Member {index+1}: {data['id']}]\n{data['text']}\n"

    reviews_text = ""
    for index, data in reviews.items():
        reviews_text += f"\n[Reviewer {index+1}: {data['id']}]\n{data['text']}\n"

    chairman_messages = _prepare_messages(
        messages,
        chairman_final_answer_prompt,
        CHAIRMAN_USER_PROMPT_TEMPLATE.format(first_opinions_section=first_opinions_text, reviews_section=reviews_text),
    )

    chairman_stream = _call_model(chairman_id, chairman_messages, stream=True)
    return chairman_stream


def validate_config(config: dict) -> None:
    """Ensure plugin configuration meets minimum requirements."""
    errors = []

    if not config.get("chairman_id"):
        errors.append("A Council Chairman must be selected.")
    members_list = [member.get(f"member") for member in config.get(f"members") if member.get(f"member")]
    if not members_list or len(members_list) < 2:
        errors.append(f"At least 2 Council Members are required (found {len(members_list) if members_list else 0}).")

    if errors:
        raise ValueError("Configuration Error:\n" + "\n".join(errors))
