import abc
import json
import queue
import time
from enum import StrEnum, auto


class Event:
    def __init__(self, type: str) -> None:
        self.type = type
        self.timestamp = int(time.time() * 1000)

    def wrap_message(self, message: str) -> str:
        return json.dumps({"time": self.timestamp, "message": message })

    @abc.abstractmethod
    def to_sse(self):
        raise NotImplementedError()

# Keep in sync with Statuses enum in app.js
class Status(StrEnum):
    NOT_INITIALIZED = auto()
    INITIALIZING = auto()
    INITIALIZED = auto()
    CONNECTING = auto()
    CONNECTED = auto()
    SHUTTING_DOWN = auto()
    ERROR_INITIALIZING = auto()
    ERROR_CONNECTING = auto()
    ERROR_RUNNING = auto()

class StatusEvent(Event):
    status: Status

    def __init__(self, status) -> None:
        super().__init__("status")
        self.status = status

    def to_sse(self) -> str:
        return super().wrap_message(self.status)

class DebugEvent(Event):
    message: str

    def __init__(self, message: str) -> None:
        super().__init__("debug")
        self.message = message

    def to_sse(self) -> str:
        return super().wrap_message(self.message)

class ErrorEvent(Event):
    message: str

    def __init__(self, message: str) -> None:
        super().__init__("slack_error")
        self.message = message

    def to_sse(self) -> str:
        return super().wrap_message(self.message)


class EventQueue:
    def __init__(self) -> None:
        self._full_queue = []
        self.listeners: list[queue.Queue[Event]] = [] 

    def listen(self) -> queue.Queue[Event]:
        q = queue.Queue()
        self.listeners.append(q)
        for event in self._full_queue:
            q.put_nowait(event)
        return q

    def announce(self, event: Event):
        self._full_queue.append(event)
        for listener in self.listeners:
            listener.put_nowait(event)

# global announcer singleton
announcer = EventQueue()
