import asyncio
import atexit
import json
from typing import cast

from agent_handler_settings import AgentHandlerSettings
from dataiku.customwebapp import get_webapp_config
from dkuslackclient.slack_manager import SlackManager
from flask import Flask, Response
from utils.logging import logger
from utils.sse_event import ErrorEvent, announcer

# Global variables
slack_manager = None

# app comes from DSS.
app = cast(Flask, app) # type: ignore # noqa: F821

@app.route("/slack-bot-status")
def status():
    def stream():
        messages = announcer.listen()
        while True:
            message = messages.get()
            yield f"event: {message.type}\ndata: {message.to_sse()}\n\n"

    # We want the X-Accel-Buffering header to tell nginx to not buffer the response. Otherwise, some events are seemingly dropped
    return Response(stream(), mimetype='text/event-stream', headers={"X-Accel-Buffering": "no"})

@app.route("/slack-bot-info")
def info():
    if slack_manager is None:
        return {}
    settings = slack_manager.get_settings()
    return settings

def setup_logging(logging_level):
    """
    Sets up the logging level using the logger.
    """
    try:
        # Set the logging level dynamically
        logger.set_level(logging_level)
        logger.info(f"Logging initialized with level: {logging_level}")
    except ValueError as e:
        # Handle invalid logging levels
        logger.error(f"Invalid logging level '{logging_level}': {str(e)}")
        raise

def cleanup():
    """Clean up resources when the application exits."""
    try:
        if slack_manager:
            loop = asyncio.get_event_loop()
            if not loop.is_closed():
                logger.info("Shutting down Slack manager...")
                loop.run_until_complete(slack_manager.cleanup())
    except Exception as e:
        logger.error(f"Failed to clean up: {str(e)}", exc_info=True)

# Register the cleanup function with atexit
atexit.register(cleanup)

# Main initialization
def init():
    """Initialize and run the Slack app."""
    global slack_manager

    logger.info("Starting Slack integration initialization...")
    config = get_webapp_config()

    # Get the logging level from the configuration, default to INFO
    logging_level = config.get("logging_level", "INFO")
    setup_logging(logging_level)

    # Get the integration mode
    # mode = config.get("mode", "socket")
    mode = "socket"  # Currently only socket mode is supported
    logger.info(f"Starting in {mode} mode")

    slack_bot_token = config.get("slack_oauth_token") # from the Install App Settings tab on the Slack App configuration page
    if not slack_bot_token:
        error_msg = "The Slack App (User or Bot User) OAuth token is missing from authentication settings."
        logger.error(error_msg)
        announcer.announce(ErrorEvent(error_msg))

    # Get the LLM ID from the configuration
    llm_id = config.get("llm_id")
    if not llm_id:
        error_msg = "No Agent or LLM selected."
        logger.error(error_msg)
        announcer.announce(ErrorEvent(error_msg))

    agent_settings = AgentHandlerSettings(
        llm_id=llm_id,
        react_on_loading=bool(config.get("react_on_loading", "False")),
        reaction_name=config.get('loading_reaction', AgentHandlerSettings.DEFAULT_LOADING_REACTION),
        conversation_context_limit=config.get('conversation_context_limit', AgentHandlerSettings.DEFAULT_CONVERSATION_CONTEXT_LIMIT),
        conversation_history_days=config.get('conversation_history_days', AgentHandlerSettings.DEFAULT_CONVERSATION_HISTORY_DAYS),
        use_custom_system_prompt=bool(config.get("use_custom_system_prompt", "False")),
        custom_system_prompt=config.get('custom_system_prompt', AgentHandlerSettings.DEFAULT_SYSTEM_PROMPT) if "use_custom_system_prompt" in config else AgentHandlerSettings.DEFAULT_SYSTEM_PROMPT
    )


    logger.info(f"Using the following agent handler settigns: {json.dumps(agent_settings.__dict__, indent=4)}")

    # Initialize based on mode
    if mode == "socket":
        # Socket mode requires an app token
        slack_app_token = config.get("slack_app_token") # from the Basic Information > App-Level Tokens tab on the Slack App configuration page
        if not slack_app_token:
            error_msg = "An App-Level Token is required for socket mode"
            logger.error(error_msg)
            announcer.announce(ErrorEvent(error_msg))

        try:
            # Initialize the SlackManager with app token for socket mode
            logger.debug("Creating SlackManager for socket mode...")
            slack_manager = SlackManager(
                slack_bot_token=slack_bot_token,
                slack_app_token=slack_app_token,
                settings=agent_settings  # Required settings dictionary with llm_id and other configuration
            )

            # Start the SlackManager in socket mode
            logger.info("Starting Slack app in socket mode...")
            slack_manager.start()
            logger.info("Slack app is running in socket mode and waiting for messages...")
        except Exception as e:
            logger.error(f"Error occurred while running Slack app in socket mode: {str(e)}", exc_info=True)
            announcer.announce(ErrorEvent(str(e)))

    elif mode == "http": # Not supported with current implementation
        raise NotImplementedError("HTTP mode is not currently supported.")

    else:
        error_msg = f"Invalid mode: {mode}. Must be either 'socket' or 'http'"
        logger.error(error_msg)
        raise ValueError(error_msg)

# Run the initialization
init()
