import asyncio
import math
import re
from datetime import datetime

from cachetools import TTLCache
from slack_sdk.web.async_client import AsyncWebClient
from slack_sdk.web.async_slack_response import AsyncSlackResponse
from utils.logging import logger
from utils.sliding_window_rate_limiter import SlidingWindowRateLimiter


class DKUSlackClient():
    """
    A client for interacting with Slack, providing functionality for handling messages,
    querying Dataiku Answers, and sending responses or reactions.
    """
    # Constants
    CHANNEL_FETCH_LIMIT = 200  # Maximum number of channels to fetch per API call
    MESSAGE_FETCH_LIMIT = 200  # Maximum number of messages to fetch per API call
    USER_FETCH_LIMIT = 100     # Maximum number of users to fetch per API call
    CACHE_TTL = 60 * 60 * 24   # 1 day in seconds
    CACHE_MAXSIZE = math.inf   # Maximum number of items in cache

    # Slack API rate limit tiers
    # https://api.slack.com/apis/rate-limits
    # How many requests per minute are allowed per tier
    TIER_1_LIMIT = 1     # Methods with the strictest rate limit (Access tier 1 methods infrequently)

    TIER_2_LIMIT = 20    # Methods with moderate rate limit
    conversation_list_rate_limiter: SlidingWindowRateLimiter = SlidingWindowRateLimiter(max_calls=TIER_2_LIMIT, window_seconds=60)  # Example: 20 calls per minute

    TIER_3_LIMIT = 50    # Methods for paginating collections of conversations or users
    conversation_replies_rate_limiter: SlidingWindowRateLimiter = SlidingWindowRateLimiter(max_calls=TIER_3_LIMIT, window_seconds=60)
    reactions_rate_limiter: SlidingWindowRateLimiter = SlidingWindowRateLimiter(max_calls=TIER_3_LIMIT, window_seconds=60)

    TIER_4_LIMIT = 100   # Methods with the loosest rate limit (Enjoy a large request quota)
    users_info_rate_limiter: SlidingWindowRateLimiter = SlidingWindowRateLimiter(max_calls=TIER_4_LIMIT, window_seconds=60)

    _slack_channel_cache = TTLCache(maxsize=CACHE_MAXSIZE, ttl=CACHE_TTL)
    _slack_user_cache = TTLCache(maxsize=CACHE_MAXSIZE, ttl=CACHE_TTL)
    _slack_channel_members_cache = TTLCache(maxsize=CACHE_MAXSIZE, ttl=CACHE_TTL)

    _slack_async_web_client: AsyncWebClient

    def __init__(self, slack_bot_token: str):
        """Initialize the Slack client with a token.

        Args:
            slack_bot_token: The Slack API bot token to use for authentication.
        """
        if not slack_bot_token:
            logger.error("Required Slack token is missing!")
            raise ValueError("Required Slack token is missing.")

        self._slack_async_web_client = AsyncWebClient(token=slack_bot_token)
        self.signature_verifier = None

        self._test()

    def _test(self):
        """Test the Slack client token."""
        # Test token using async client directly
        response = asyncio.run(self._slack_async_web_client.auth_test())
        if not response["ok"]:
            logger.error("Token test failed: %s", response.get("error"))
            raise ValueError(f"Token test failed: {response.get('error')}")

        if response.get("team_id") is None:
            logger.error("Token test was successful, but we couldn't connect to a Slack workspace. Is someone already using the credentials?")
            raise ValueError("No workspace found after auth test")

        # Log authentication information
        logger.info("Token test successful")
        logger.info("Workspace: %s (ID: %s)", response.get("team"), response.get("team_id"))
        logger.info("User: %s (ID: %s)", response.get("user"), response.get("user_id"))
        logger.info("URL: %s", response.get("url"))

        self.workspace_name = response.get("team")

    @property
    def slack_async_web_client(self):
        """Get the Slack AsyncWebClient instance."""
        return self._slack_async_web_client

    async def get_bot_info(self) -> tuple[str, str]:
        """Get the bot's ID and name."""
        logger.debug("Fetching app authentication info...")
        # Use the DKUSlackClient's AsyncWebClient directly
        auth_info = await self.slack_async_web_client.auth_test()
        bot_id = auth_info["user_id"]
        bot_name = auth_info["user"]
        return bot_id, bot_name

    def _cache_user_info(self, user_id: str, user_info: dict):
        """
        Process and cache user information.

        :param user_id: The user's ID
        :param user_info: The user information from Slack API
        :return: Tuple containing (user_id, display_name, email)
        """
        username = user_info.get("real_name", "Unknown User")
        display_name = user_info.get("profile", {}).get("display_name", username)
        email = user_info.get("profile", {}).get("email", "No email found")

        # Cache the complete user info
        self._slack_user_cache[user_id] = {
            "name": display_name or username,
            "email": email,
            "timestamp": datetime.now()
        }
        return user_id, display_name or username, email

    async def _get_user_by_id(self, user_id: str):
        """
        Get user information from Slack by user ID.
        Caches the results for future lookups.

        :param user_id: Slack user ID to fetch information for
        :return: Tuple containing (user_id, display_name, email) or (None, None, None) if not found
        """
        logger.info(f"Getting user by ID {user_id}")
        # Check cache first
        cached_user = self._slack_user_cache.get(user_id)
        if cached_user:
            logger.debug(f"Using cached user info for {cached_user['name']} ({cached_user['email']})")
            return user_id, cached_user["name"], cached_user["email"]

        # If not in cache, get user info
        logger.debug(f"Cached user info was not found for user {user_id}, fetching from Slack API")
        await self.users_info_rate_limiter.await_for_slot()
        response = await self._slack_async_web_client.users_info(user=user_id)
        if response:
            return self._cache_user_info(user_id, response["user"])
        return None, None, None

    async def get_channel_name(self, channel_id: str):
        if cached_name := self._slack_channel_cache.get(channel_id):
            return cached_name

        response = await self._slack_async_web_client.conversations_info(channel=channel_id)
        self._slack_channel_cache[channel_id] = response["channel"]["name_normalized"]
        return response["channel"]["name_normalized"]

    @reactions_rate_limiter.decorator
    async def send_reaction(self, channel_id: str, event_timestamp: str, reaction_name: str):
        """
        Sends a reaction emoji to a Slack message.

        :param channel_id: ID of the channel where the message was sent.
        :param reaction_name: Name of the reaction emoji.
        :param event_timestamp: Timestamp of the Slack message.
        """
        response = await self._slack_async_web_client.reactions_add(channel=channel_id, name=reaction_name, timestamp=event_timestamp)
        logger.info(f"Successfully sent reaction '{reaction_name}' to message {event_timestamp} in channel {channel_id}")
        return response

    @reactions_rate_limiter.decorator
    async def remove_reaction(self, channel_id: str, event_timestamp: str, reaction_name: str):
        """
        Sends a reaction emoji to a Slack message.

        :param channel_id: ID of the channel where the message was sent.
        :param reaction_name: Name of the reaction emoji.
        :param event_timestamp: Timestamp of the Slack message.
        """
        response = await self._slack_async_web_client.reactions_remove(channel=channel_id, name=reaction_name, timestamp=event_timestamp)
        logger.info(f"Successfully removed reaction '{reaction_name}' from message {event_timestamp} in channel {channel_id}")
        return response

    async def _add_user_info_to_messages(self, messages: list[dict]) -> list[dict]:
        """
        Add user information (user_name, user_email) to messages based on user IDs.
        For replies, also process reply_users field and add corresponding user info.
        Also resolves user mentions in message text (e.g., <@UF9QWN8RJ>).

        :param messages: List of message objects from Slack API
        :return: List of messages with added user information
        """
        logger.info(f"Adding user information to {len(messages)} messages")

        # Create a set of all user IDs that need to be resolved
        user_ids_to_resolve = set()

        # Regular expression to find user mentions in text
        user_mention_pattern = r'<@([A-Z0-9]+)>'

        # Collect all user IDs from messages, replies, and text mentions
        for message in messages:
            # Add sender user ID
            if "user" in message:
                user_ids_to_resolve.add(message["user"])

            # Add reply user IDs if present
            if "reply_users" in message:
                user_ids_to_resolve.update(message["reply_users"])

            # Find user mentions in message text
            if "text" in message:
                mentions = re.findall(user_mention_pattern, message["text"])
                user_ids_to_resolve.update(mentions)

        # Create a mapping of user_id to user info
        user_info_map = {}

        # Create tasks for all user IDs to get user information in parallel
        tasks = [self._get_user_by_id(uid) for uid in user_ids_to_resolve]
        results = await asyncio.gather(*tasks)

        # Create a mapping from the results
        for i, uid in enumerate(user_ids_to_resolve):
            user_id, user_name, user_email = results[i]
            if user_id:  # Skip None results
                user_info_map[uid] = {
                    "user_id": user_id,
                    "user_name": user_name,
                    "user_email": user_email
                }

        # Add user information to each message
        for message in messages:
            # Add sender information
            if "user" in message and message["user"] in user_info_map:
                user_info = user_info_map[message["user"]]
                message["user_name"] = user_info["user_name"]
                message["user_email"] = user_info["user_email"]

            # Add parent user information if present
            if "parent_user_id" in message and message["parent_user_id"] in user_info_map:
                parent_user_info = user_info_map[message["parent_user_id"]]
                message["parent_user_name"] = parent_user_info["user_name"]
                message["parent_user_email"] = parent_user_info["user_email"]

            # Add reply users information
            if "reply_users" in message:
                message["reply_users_info"] = []
                for reply_user_id in message["reply_users"]:
                    if reply_user_id in user_info_map:
                        message["reply_users_info"].append(user_info_map[reply_user_id])

            # Process user mentions in message text
            if "text" in message:
                # Find all user mentions
                mentions = re.findall(user_mention_pattern, message["text"])
                if mentions:
                    # Create a list to store mention information
                    message["mentions"] = []
                    for user_id in mentions:
                        if user_id in user_info_map:
                            message["mentions"].append(user_info_map[user_id])

                    # Replace mentions in text with user names
                    for user_id in mentions:
                        if user_id in user_info_map:
                            user_name = user_info_map[user_id]["user_name"]
                            message["text"] = message["text"].replace(
                                f"<@{user_id}>", 
                                f"@{user_name}"
                            )

        logger.info(f"Successfully added user information to {len(messages)} messages")
        return messages

    async def fetch_thread_replies(self, channel_id: str, thread_ts: str, latest_message_ts: str | None = None, context_days: int | None = None, resolve_users=True) -> tuple[list[dict], str | None]:
        """
        Fetch replies for a specific thread using conversations.replies.

        :param channel_id: ID of the channel containing the thread
        :param thread_ts: Timestamp of the parent message
        :param latest_message_ts: Timestamp of the latest message to fetch (optional)
        :param resolve_users: Whether to resolve user IDs to usernames and emails (default: True)
        :return: Tuple of (replies, error) where error is None if successful, or error message if failed
        """
        logger.info(f"Fetching replies for thread {thread_ts} in channel {channel_id}")
        replies: list[dict] = []
        next_cursor: str | None = None
        oldest_ts: str | None = None
        if context_days and latest_message_ts:
            oldest_ts = str(float(latest_message_ts) - (context_days * 24 * 60 * 60 * 100))

        while True:
            await self.conversation_replies_rate_limiter.await_for_slot()
            response = await self._slack_async_web_client.conversations_replies(
                channel=channel_id,
                ts=thread_ts,
                latest=latest_message_ts,
                oldest=oldest_ts,
                inclusive=True,
                limit=self.MESSAGE_FETCH_LIMIT,
                cursor=next_cursor
            )

            if not response["ok"]:
                error_msg = f"Failed to fetch thread replies: {response.get('error')}"
                logger.error(error_msg)
                return replies, error_msg

            replies += response.get("messages", [])

            next_cursor = response.get("response_metadata", {}).get("next_cursor")
            if next_cursor is None:
                break

        replies: list[dict] = response.get("messages", [])

        # Add user information to all replies if requested
        if resolve_users:
            logger.info("Resolving user IDs to usernames and emails...")
            replies = await self._add_user_info_to_messages(replies)
        else:
            logger.info("Skipping user ID resolution as requested")

        logger.info(f"Successfully fetched {len(replies)} replies for thread {thread_ts}")
        return replies, None

    # Obscure rate limiting for posting messages https://docs.slack.dev/reference/methods/chat.postMessage/#rate_limiting
    async def post_thread_message(self, channel_id: str, thread_ts: str, text: str, blocks=None) -> AsyncSlackResponse:
        """
        Post a message in a Slack thread.

        :param channel_id: ID of the channel where the thread is located.
        :param thread_ts: Timestamp of the parent message to reply to.
        :param text: Text content of the message.
        :param blocks: Optional Slack blocks for rich formatting.
        :return: The response from Slack API.
        """
        # Use files_upload_v2 for file handling in the future
        response = await self._slack_async_web_client.chat_postMessage(
            channel=channel_id,
            thread_ts=thread_ts,
            text=text,
            blocks=blocks
        )
        logger.info(f"Successfully posted message to thread {thread_ts} in channel {channel_id}")
        return response

    async def set_bot_name(self, bot_name: str):
        """
        Set the bot's name.
        :param bot_name: The new name for the bot.
        """
        bot_id, bot_name = await self.get_bot_info()
        await self._slack_async_web_client.users_profile_set(
            user=bot_id,
            profile={"display_name": bot_name},
        )
