import base64

import dataiku
import requests
from agent_handler_settings import AgentHandlerSettings
from dataikuapi.dss.llm import DSSLLMCompletionQuery
from dkuslackclient.dku_slack_client import DKUSlackClient
from utils.logging import logger


class SlackEventHandler:
    """
    Handles Slack event processing logic, separate from the transport mechanism.
    This class contains the business logic for processing Slack events.
    """

    # Constants
    MENTION_WITHOUT_TEXT = """
Hi there! You didn't provide a message with your mention.
Mention me again in this thread so that I can help you out!
"""
    DEFAULT_SYSTEM_PROMPT = """You are a versatile AI assistant. Your name is {bot_name}.
Help users with writing, coding, task management, advice, project management, and any other needs.
Provide concise, relevant assistance tailored to each request.
Note that context is sent in order of the most recent message last.
Do not respond to messages in the context, as they have already been answered.
Be professional and friendly.
Don't ask for clarification unless absolutely necessary.
Don't ask questions in your response.
Don't use user names in your response.
Respond using Slack markdown.
"""

    def __init__(self, bot_id: str, bot_name: str, slack_client: DKUSlackClient, settings: AgentHandlerSettings, slack_bot_token: str):
        """
        Initialize the SlackEventHandler.

        Args:
            bot_id: The bot's user ID (optional)
            bot_name: The bot's name (optional)
            slack_client: SlackClient instance for API calls
            settings: settings for the handling to the agent actions.
        """
        logger.debug("Initializing SlackEventHandler...")
        self.bot_id = bot_id
        self.bot_name = bot_name
        self.slack_client = slack_client
        self.slack_bot_token = slack_bot_token
        self.settings = settings

        # LLM info cache
        self._llm_name = None
        self._llm_type = None

        # Initialize LLM client if llm_id is provided
        logger.debug(f"Initializing LLM client with ID: {self.settings.llm_id}")
        self.llm_client = dataiku.api_client().get_default_project().get_llm(self.settings.llm_id)
        logger.info(f"LLM client initialized for {self.settings.llm_id}")

    def _get_file_data_from_url(self, url: str) -> str:
            HEADERS = {'Authorization': 'Bearer ' + self.slack_bot_token}
            return base64.b64encode(requests.get(url, headers=HEADERS).content).decode("utf8")

    def _slack_messages_to_dss_query(self, slack_messages: list[dict]) -> DSSLLMCompletionQuery:
        """
        Convert Slack message format to the DSS query format.

        :param slack_bot_token: Slack bot token for file access
        :param messages: List of Slack messages
        :return: DSSLLMCompletionQuery object
        """
        query: DSSLLMCompletionQuery = self.llm_client.new_completion()
        if self.settings.use_custom_system_prompt:
            query = query.with_message(message=self.settings.custom_system_prompt, role="system")

        for message in slack_messages:
            role = "assistant" if message.get("user") == self.bot_id else "user"
            dss_message = query.new_multipart_message(role)
            text = message.get("text")
            if text:
                dss_message.with_text(text)

            files = message.get("files", [])
            for file in files:
                if "image" in file.get("mimetype", ""):
                    image_url = file.get("url_private")
                    if image_url:
                        image_base64 = self._get_file_data_from_url(image_url)
                        dss_message.with_inline_image(image_base64)

            dss_message.add()

        return query


    def get_llm_info(self):
        """
        Get LLM information (name and type) from Dataiku API.
        Caches results for subsequent calls.

        Returns:
            tuple: A tuple containing (llm_name, llm_type)
        """
        # Return cached values if available
        if self._llm_name is not None and self._llm_type is not None:
            return self._llm_name, self._llm_type

        # Default values
        llm_name = "Unknown LLM"
        llm_type = "UNKNOWN"

        # Attempt to retrieve LLM info from Dataiku API
        client = dataiku.api_client()
        project = client.get_default_project()
        llm_list = project.list_llms()

        for llm in llm_list:
            if llm.id == self.settings.llm_id: # type: ignore
                llm_name = llm.get('friendlyName', 'Unknown LLM') # type: ignore
                llm_type = llm.get('type', 'UNKNOWN') # type: ignore
                logger.debug(f"Found LLM name for {self.settings.llm_id}: {llm_name}, type: {llm_type}")
                break

        # Cache the values
        self._llm_name = llm_name
        self._llm_type = llm_type

        return llm_name, llm_type

    async def handle_user_input(self, event_data: dict, is_mention=False):
        """
        Common handler for both messages and mentions.

        Args:
            event_data: The event data (message or mention)
            client: Slack WebClient for API calls
            is_mention: Whether this is a mention event

        Returns:
            None
        """
        event_type = "mention" if is_mention else "message"
        logger.debug(f"Handling {event_type} event: {event_data}")

        # Skip messages from the bot itself
        user_id = event_data.get("user")
        if user_id == self.bot_id:
            logger.debug(f"Skipping {event_type} from bot itself")
            return

        # Skip messages from any bot
        if event_data.get("bot_id") is not None:
            logger.debug(f"Skipping {event_type} from another bot")
            return

        # Get channel and thread info
        channel = event_data["channel"]

        # Check if the message is from a bot (not our bot)
        is_from_bot = event_data.get("bot_id") is not None and event_data.get("bot_id") != self.bot_id

        # If thread_ts is not present this is the first message, use ts to start a new thread
        thread_ts = event_data.get("thread_ts", event_data["ts"])

        message_ts = event_data["ts"]

        # Get the text of the message
        text = event_data.get("text", "")

        # Remove bot mention from text if present
        if self.bot_id:
            text = text.replace(f"<@{self.bot_id}>", "").strip()

        # Handle case where user only mentioned the bot without text
        if is_mention and not text:
            logger.info("User mentioned bot without providing text")
            reply = await self.slack_client.post_thread_message(
                channel_id=channel,
                thread_ts=thread_ts,
                text=self.MENTION_WITHOUT_TEXT,
            )
            return

        react_on_loading = self.settings.react_on_loading
        # Send loading reaction
        if react_on_loading:
            try:
                await self.slack_client.send_reaction(channel, message_ts, self.settings.reaction_name)
            except Exception as e:
                logger.error(f"Error adding loading reaction: {str(e)}", exc_info=True)
                react_on_loading = False  # Disable reaction removal if adding failed

        # Generate response using LLM
        response = await self.generate_response(channel, thread_ts, message_ts)
        text_to_send = response.text if response.success else "I'm sorry, I couldn't generate a response at this time."
        if not response.success:
            logger.error(f"LLM response generation failed: {response._raw.get('errorMessage')}")

        reply = await self.slack_client.post_thread_message(
            channel_id=channel,
            thread_ts=thread_ts,
            text=text_to_send, # type: ignore
        )

        if reply and reply.get("ok"):
            logger.info(f"Posted response message in channel {channel} thread {thread_ts}")
        else:
            logger.error(f"Failed to post response message in channel {channel} thread {thread_ts}: {reply}")

        # Remove loading reaction
        if react_on_loading:
            await self.slack_client.remove_reaction(channel, message_ts, self.settings.reaction_name)

    async def handle_app_home_event(self, event, client):
        """
        Handle an app_home_opened event from Slack.

        Args:
            event: The app_home_opened event data from Slack
            client: Slack WebClient for API calls
        """
        logger.debug(f"Handling app home event: {event}")
        user_id = event.get("user")
        view = self.generate_home_view()

        try:
            await self.slack_client._slack_async_web_client.views_publish(user_id=user_id, view=view)
            logger.info(f"Published home view for user {user_id}")
        except Exception as e:
            logger.error(f"Error publishing home view: {str(e)}", exc_info=True)

    async def generate_response(self, channel: str, thread_ts: str, message_ts: str):
        """
        Generate a response using the LLM.

        Args:
            channel: The channel ID
            thread_ts: The thread timestamp
            text: The text to respond to
            event_data: The original event data

        Returns:
            dict: Response data including text and blocks
        """

        # Get conversation history from Slack
        thread_messages, error = await self.slack_client.fetch_thread_replies(
            channel_id=channel,
            thread_ts=thread_ts,
            latest_message_ts=message_ts,
            context_days=self.settings.conversation_history_days
        )

        if len(thread_messages) > self.settings.conversation_context_limit:
            logger.info(f"Trimming thread of {len(thread_messages)} messages to {self.settings.conversation_context_limit}.")
            thread_messages = thread_messages[-self.settings.conversation_context_limit:]

        if error:
            raise Exception(f"Error fetching thread replies: {error}")

        completion_query = self._slack_messages_to_dss_query(thread_messages)
        return completion_query.execute()

    def generate_home_view(self):
        """
        Generate the App Home view content.

        Returns:
            dict: The view object for the App Home
        """

        # Get LLM information
        llm_name, llm_type = self.get_llm_info()
        # when llm_type is SAVED_MODEL_AGENT, agent is already part of the llm_name so no need to add anything
        header_display_llm = f'{llm_name}{"" if llm_type == "SAVED_MODEL_AGENT" else " augmented LLM" if llm_type == "RETRIEVAL_AUGMENTED" else " LLM"}'
        display_model_type = "Agent" if llm_type == "SAVED_MODEL_AGENT" else "augmented LLM" if llm_type == "RETRIEVAL_AUGMENTED" else "LLM"
        blocks = [
            {
                "type": "header",
                "text": {"type": "plain_text", "text": f"Welcome to the Dataiku {header_display_llm} bot!"},
            },
            {
                "type": "section",
                "text": {
                    "type": "mrkdwn",
                    "text": f"This is a Slack bot to interact with the *{display_model_type}* hosted on Dataiku DSS.",
                },
            },
        ]

        # Add usage section
        blocks.append(
            {
                "type": "section",
                "text": {
                    "type": "mrkdwn",
                    "text": (
                        "*How to Use:*\n• Send me a direct message\n"
                        f"• Mention me in a channel with @{self.bot_name or 'Bot'}"
                    ),
                },
            }
        )

        return {"type": "home", "blocks": blocks} 
