import asyncio
import threading

from agent_handler_settings import AgentHandlerSettings
from slack_bolt import Say
from slack_bolt.adapter.sanic import AsyncSlackRequestHandler
from slack_bolt.adapter.socket_mode.async_handler import AsyncSocketModeHandler
from slack_bolt.app.async_app import AsyncApp
from utils.logging import logger
from utils.sse_event import (DebugEvent, ErrorEvent, Status, StatusEvent,
                             announcer)

from .dku_slack_client import DKUSlackClient
from .slack_event_handler import SlackEventHandler


class SlackManager:
    """
    Manages the Slack connection and routes events to the SlackEventHandler.
    Supports both socket mode and HTTP endpoint mode.
    """

    app: AsyncApp
    socket_mode_handler: AsyncSocketModeHandler
    status: Status = Status.NOT_INITIALIZED

    def set_status(self, status: Status):
        if status != self.status:
            self.status = status
            announcer.announce(StatusEvent(status))

    def __init__(self, slack_bot_token: str, slack_app_token: str, settings: AgentHandlerSettings):
        """
        Initialize the SlackManager.

        Args:
            slack_bot_token: The bot user OAuth token
            slack_app_token: The app-level token for Socket Mode (required for socket mode)
            settings: Dictionary of required settings to customize behavior (contains llm_id and other configuration)
        """
        logger.debug("Initializing SlackManager...")
        self.slack_bot_token = slack_bot_token
        self.slack_app_token = slack_app_token
        self.mode = "socket"

        # Create settings dictionary with all configuration parameters
        self.settings = settings

        self.set_status(Status.INITIALIZING)

        try:
            # Create a DKUSlackClient instance
            self.slack_client_instance = DKUSlackClient(slack_bot_token)

            # Initialize Slack App instance
            self.app = AsyncApp(token=slack_bot_token)
            self.request_handler = None
            self.thread = None

            # Fetch bot info
            logger.debug("Fetching app authentication info...")
            bot_id, bot_name = asyncio.run(self.slack_client_instance.get_bot_info())
            logger.info(f"App initialized with User ID: {bot_id} and Name: {bot_name}")

            # Create the event handler with bot info and settings
            self.event_handler = SlackEventHandler(
                bot_id,
                bot_name,
                slack_client=self.slack_client_instance,
                settings=self.settings,
                slack_bot_token=slack_bot_token
            )

            # Set up event handlers
            self._setup_listeners()
            logger.info(f"SlackManager initialized in {self.mode} mode with LLM ID: {self.settings.llm_id}")
            self.set_status(Status.INITIALIZED)
        except:
            self.set_status(Status.ERROR_INITIALIZING)
            raise

    def _setup_listeners(self):
        """Set up event listeners for the Slack app."""
        logger.debug("Setting up event listeners...")

        # Handle message events
        @self.app.event("message")
        async def direct_message_listener(say: Say, event: dict):
            try:
                logger.debug(f"Received message event: {event}")
                announcer.announce(DebugEvent("Received direct message"))
                # Delegate handling to the event handler
                await self.event_handler.handle_user_input(event, is_mention=False)
            except Exception as e:
                self.set_status(Status.ERROR_RUNNING)
                announcer.announce(ErrorEvent(str(e)))

        # Handle app mention events
        @self.app.event("app_mention")
        async def app_mention_listener(say: Say, event: dict):
            try:
                logger.debug(f"Received app mention event: {event}")
                announcer.announce(DebugEvent("Received app mention"))
                # Delegate handling to the event handler
                await self.event_handler.handle_user_input(event, is_mention=True)
            except Exception as e:
                self.set_status(Status.ERROR_RUNNING)
                announcer.announce(ErrorEvent(str(e)))

        # Handle app home opened events
        @self.app.event("app_home_opened")
        async def app_home_opened_listener(event, client):
            try:
                logger.debug(f"Received app home opened event: {event}")
                announcer.announce(DebugEvent("Received home opened"))
                # Delegate handling to the event handler
                await self.event_handler.handle_app_home_event(event, client)
            except Exception as e:
                self.set_status(Status.ERROR_RUNNING)
                announcer.announce(ErrorEvent(str(e)))

    def start(self):
        """Start the Slack integration based on the configured mode."""
        if self.mode == "socket":
            return self._start_socket_mode()
        else:
            return self._prepare_http_mode()

    def _start_socket_mode(self):
        """Start the Socket Mode handler in a separate thread."""
        if not self.slack_app_token:
            error_msg = "Cannot start in socket mode without an app token"
            logger.error(error_msg)
            raise ValueError(error_msg)

        try:
            logger.info("Starting Slack in socket mode...")

            # Initialize the Socket Mode handler
            threading.Thread(target=self._run_socket_handler, name="SlackSocketModeThread", daemon=True).start()
            return True
        except Exception as e:
            self.set_status(Status.ERROR_CONNECTING)
            logger.error(f"Failed to start socket mode: {str(e)}", exc_info=True)
            raise

    def _run_socket_handler(self):
        """Run the socket mode handler (called in a thread)."""
        try:
            self.set_status(Status.CONNECTING)
            logger.debug(f"Socket mode handler thread starting (Thread ID: {threading.get_ident()}, Name: {threading.current_thread().name})")
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)
            self.socket_mode_handler = AsyncSocketModeHandler(self.app, self.slack_app_token, loop=loop)

            # Slack SDK doesn't give us any other way to check that the app-token is valid. If it's invalid, the connect() method in
            # aiohttp/__init__.py will loop infinitely. One way to check is the session id: it will exist only if we're correctly connected
            # to a workspace.
            async def session_check():
                for _ in range(10):
                    session_id = await self.socket_mode_handler.client.session_id()
                    if session_id is not None and len(session_id) > 0:
                        logger.debug("Created a session to Slack servers")
                        self.set_status(Status.CONNECTED)
                        return
                    await asyncio.sleep(1)

                logger.error("Can't connect to Slack servers.")
                self.set_status(Status.ERROR_CONNECTING)
                announcer.announce(ErrorEvent("There was an issue connecting to the Slack servers through the WebSocket."))
            asyncio.ensure_future(session_check(), loop=loop)

            # Set thread name in logs for better traceability
            loop.run_until_complete(self.socket_mode_handler.start_async())
            loop.close()
        except Exception as e:
            announcer.announce(ErrorEvent(str(e)))
            logger.error(f"Error in socket mode thread: {str(e)}", exc_info=True)
            if self.status == Status.CONNECTED:
                self.set_status(Status.ERROR_RUNNING)
            # Re-raise the exception to ensure it's visible in the main process
            raise

    def _prepare_http_mode(self):
        """Prepare for HTTP mode by creating a request handler."""
        logger.info("Preparing Slack for HTTP mode...")
        self.request_handler = AsyncSlackRequestHandler(self.app)
        logger.info("Slack HTTP request handler initialized")
        return self.request_handler

    async def cleanup(self):
        """Clean up resources."""
        logger.info("Starting cleanup process...")
        self.set_status(Status.SHUTTING_DOWN)

        try:
            if self.socket_mode_handler and self.mode == "socket":
                logger.debug("Closing socket mode handler...")
                await self.socket_mode_handler.close_async()
                logger.info("Socket mode handler closed")
        except Exception as e:
            logger.error(f"Error closing socket mode handler: {str(e)}", exc_info=True)

        logger.info("Cleanup process completed")

    def handle_http_request(self, request):
        """
        Handle an HTTP request using the Flask adapter.

        Args:
            request: Flask request object

        Returns:
            Flask response
        """
        if self.request_handler:
            return self.request_handler.handle(request)
        else:
            logger.error("No request handler available")
            raise ValueError("Slack manager not initialized for HTTP mode")

    def get_settings(self):
        llm_name, _ = self.event_handler.get_llm_info()
        return {
            "llm_name": llm_name,
            "workspace": self.slack_client_instance.workspace_name,
        }
