import asyncio
import time
from collections import deque
from functools import wraps


class SlidingWindowRateLimiter:
    max_calls: int
    window: int
    calls: deque = deque()
    lock = asyncio.Lock()

    def __init__(self, max_calls: int, window_seconds: int):
        if max_calls <= 1 or window_seconds <= 1:
            raise ValueError("max_calls and window_seconds must be greater than 1")
        self.max_calls = max_calls
        self.window = window_seconds

    async def allow_request(self) -> bool:
        now = time.time()
        async with self.lock:
            # Remove timestamps outside the window
            while self.calls and self.calls[0] <= now - self.window:
                self.calls.popleft()

            # we can allow the request if we have not reached max_calls in the window
            if len(self.calls) < self.max_calls:
                self.calls.append(now)
                return True
            else:
                return False

    async def await_for_slot(self):
        while not await self.allow_request():
            async with self.lock:
                now = time.time()
                sleep_for = self.window - (now - self.calls[0]) if self.calls else 0 # self.calls could be empty if another coroutine modified it
            if sleep_for > 0:
                await asyncio.sleep(sleep_for)

    def decorator(self, fn):
        @wraps(fn)
        async def wrapper(*args, **kwargs):
            await self.await_for_slot()
            return await fn(*args, **kwargs)
        return wrapper