let isConnected = false;
let firstError = true; // Used to switch to the setup tab if there is an error somewhere
let sse;
let retryId;

// Keep in sync with Status enum in sse_event.py
const Statuses = {
    NOT_INITIALIZED: {
        message: "not initialized",
        className: "orange",
    },
    INITIALIZING: {
        message: "initializing",
        className: "orange",
    },
    INITIALIZED: {
        message: "initialized",
        className: "orange",
    },
    CONNECTING: {
        message: "connecting",
        className: "orange",
    },
    CONNECTED: {
        message: "connected",
        className: "green",
    },
    ERROR_INITIALIZING: {
        message: "error initializing",
        className: "red",
    },
    ERROR_CONNECTING: {
        message: "error connecting",
        className: "red",
    },
    ERROR_RUNNING: {
        message: "error while running",
        className: "red",
    },
    BACKEND_NOT_STARTED: {
        message: "backend not started",
        className: "orange",
    },
    UNKNOWN_ERROR: {
        message: "unknown error",
        className: "red",
    }
};

// #region: ui setup
function setupTab(buttonA, buttonB, contentA, contentB) {
    buttonA.addEventListener("click", () => {
        contentA.classList.remove("hidden");
        contentB.classList.add("hidden");
        buttonA.classList.add("active");
        buttonB.classList.remove("active");
    });
    buttonB.addEventListener("click", () => {
        contentA.classList.add("hidden");
        contentB.classList.remove("hidden");
        buttonA.classList.remove("active");
        buttonB.classList.add("active");
    });
}

const eventsTabButton = document.getElementById("tab-events-button");
const setupTabButton = document.getElementById("tab-setup-button");
const eventsContent = document.getElementById("events-tab");
const setupContent = document.getElementById("setup-tab");
setupTab(eventsTabButton, setupTabButton, eventsContent, setupContent);

const setupQuickStartButton = document.getElementById("setup-quick-start-button");
const setupStandardButton = document.getElementById("setup-standard-button");
const setupQuickStartContent = document.getElementById("setup-quick-start-content");
const setupStandardContent = document.getElementById("setup-standard-content");
setupTab(setupQuickStartButton, setupStandardButton, setupQuickStartContent, setupStandardContent);

document.getElementById("copy-button").addEventListener("click", async () => {
    try {
        await navigator.clipboard.writeText(
`{
    "display_information": {
        "name": "Dataiku Agent App",
        "description": "App to interact with a Dataiku Agent through a bot.",
        "background_color": "#1c2a38",
        "long_description": "I'm a bot to interact with a Dataiku Agent. You can talk to me in direct messages or mention me in a channel I'm part of. I will reply in the same thread or start one and consider the thread as part of our conversation."
    },
    "features": {
        "app_home": {
            "home_tab_enabled": true,
            "messages_tab_enabled": true,
            "messages_tab_read_only_enabled": false
        },
        "bot_user": {
            "display_name": "Dataiku Agent",
            "always_online": true
        }
    },
    "oauth_config": {
        "scopes": {
            "bot": [
                "app_mentions:read",
                "channels:history",
                "chat:write",
                "files:read",
                "files:write",
                "im:history",
                "incoming-webhook",
                "mpim:history",
                "reactions:write",
                "users:read"
            ]
        }
    },
    "settings": {
        "event_subscriptions": {
            "bot_events": [
                "app_home_opened",
                "app_mention",
                "message.im"
            ]
        },
        "interactivity": {
            "is_enabled": true
        },
        "org_deploy_enabled": false,
        "socket_mode_enabled": true,
        "token_rotation_enabled": false
    }
}`);
        alert("Manifest copied to clipboard!");
    } catch (error) {
        console.error(error.message);
        alert("Can't copy to clipboard. You can select the text and copy it manually.");
    }
});
// #endregion

function pushDebugEvent(event, dedup = false) {
    const newDebugEvents = document.getElementById("events-tab");
    const template = document.getElementById("event-line-template");
    const clone = document.importNode(template.content, true);
    const timestampElement = clone.querySelectorAll(".event-timestamp");
    const textElement = clone.querySelectorAll(".event-text");
    timestampElement[0].innerText = new Date(event.time).toLocaleString();
    textElement[0].innerText = event.message;

    if (newDebugEvents.children.length === 0 || !dedup) {
        newDebugEvents.appendChild(clone);
    } else {
        const lastChild = newDebugEvents.children[newDebugEvents.children.length - 1];
        if (lastChild.querySelectorAll(".event-text")[0].innerText === event.message) {
            const repeatElement = lastChild.querySelectorAll(".event-repeat");
            if (repeatElement[0].innerText === "") repeatElement[0].innerText = "1";
            repeatElement[0].innerText = parseInt(repeatElement[0].innerText) + 1;
        } else {
            newDebugEvents.appendChild(clone);
        }
    }
}

function pushDedupError(htmlMessage) {
    const container = document.getElementById("error-container");
    for (const child in container.children) {
        if (child.innerHTML === htmlMessage) {
            return;
        }
    }

    const template = document.getElementById("error-template");
    const clone = document.importNode(template.content, true);
    const textElement = clone.querySelectorAll(".error-text");
    textElement[0].innerHTML = htmlMessage;
    container.appendChild(clone);
}

function setStatus(status) {
    const statusElement = document.getElementById("current-status");
    statusElement.innerText = status.message;
    statusElement.className = status.className;
}

function setupSSE() {
    retryId = null;
    const backendUrl = dataiku.getWebAppBackendUrl("/slack-bot-status");
    sse = new EventSource(backendUrl);
    sse.onopen = (event) => {
        clearTimeout(retryId);
        const errorContainer = document.getElementById("error-container");
        errorContainer.innerHTML = "";
        isConnected = true;
    }

    sse.addEventListener("status", async (event) => {
        const parsedEvent = JSON.parse(event.data);
        setStatus(Statuses[parsedEvent.message.toUpperCase()]);
        pushDebugEvent({
            "time": parsedEvent.time,
            "message": "Received status '" + parsedEvent.message + "' from webapp backend"
        });

        if (parsedEvent.message == "connected") {
            const response = await fetch(dataiku.getWebAppBackendUrl("/slack-bot-info"));
            if (!response.ok) {
                pushDedupError("Couldn't fetch agent information.")
                return;
            }
            const data = await response.json();
            const llmNameElement = document.getElementById("llm-name");
            llmNameElement.innerText = data.llm_name;
            const workspaceElement = document.getElementById("workspace");
            workspaceElement.innerText = data.workspace;
        }
    })

    sse.addEventListener("debug", (event) => {
        pushDebugEvent(JSON.parse(event.data));
    })

    sse.addEventListener("slack_error", (event) => {
        const parsedEvent = JSON.parse(event.data);
        pushDebugEvent({
            "time": parsedEvent.time,
            "message": "Received error from webapp backend: " + parsedEvent.message
        });
        pushDedupError(parsedEvent.message);
    })

    sse.onerror = (event) => {
        isConnected = false;

        if (firstError) {
            eventsTabButton.classList.remove("active");
            setupTabButton.classList.add("active");
            eventsContent.classList.add("hidden");
            setupContent.classList.remove("hidden");
            firstError = false;
        }

        if (retryId === null) {
            retryId = setTimeout(() => {
                setupSSE();
            }, 1000);
        }

        pushDebugEvent({
            message: "Backend seems not started, retrying in 1 second...",
            time: Date.now(),
        }, true)

        // Hack. The generated HTML in AbstractStandardWebAppHandler doesn't replace the backendUrlPrefix variable if there is no backend.
        // In our case, it's when the backend is not started and we open the view page.
        if (backendUrl.includes("${backendUrlPrefix}")) {
            setStatus(Statuses.BACKEND_NOT_STARTED);
        } else {
            setStatus(Statuses.UNKNOWN_ERROR);

            // Would be useful for debug if something is wrong
            console.log("Detected backend URL:", backendUrl);
            console.log("Received event:", event);

            pushDedupError("Can't communicate to the webapp backend.");
        }
    }
}

setupSSE();

console.log("Backend url", getWebAppBackendUrl("/slack-bot-status"));
