import time
import logging
import boto3
import json

# Boto is too verbose in DEBUG
logging.getLogger('boto3').setLevel(logging.INFO)
logging.getLogger('botocore').setLevel(logging.INFO)

def invoke_agent(
    aws_auth,
    aws_region,
    agent_id,
    alias_id,
    user_message, 
    conversation_id,
    stream_interval=5,
    enable_streaming=True,
    enable_trace=True,
    inital_retry_delay=1, # seconds
    max_retry_attempts=3
):
    """
    Invoke agent (retry with exponential backoff).
    API docs: https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html
    """

    
    bedrock_client=boto3.client(
        service_name="bedrock-agent-runtime",
        region_name=aws_region,
        **aws_auth
    )

    retries = 0
    response = None
    current_delay = inital_retry_delay
    error = None
    
    while retries <= max_retry_attempts:
        try:
            logging.info("Invoking Bedrock Agent with ID %s, alias %s, conversation_id %s" % (agent_id, alias_id, conversation_id))
            response = bedrock_client.invoke_agent(
                agentId=agent_id,
                agentAliasId=alias_id,
                enableTrace=enable_trace,
                sessionId = conversation_id,
                inputText=user_message,
                streamingConfigurations = {
                    "applyGuardrailInterval" : stream_interval,
                    "streamFinalResponse" : enable_streaming
                }
            )
            break
        except Exception as e:
            logging.exception("Error invoking Bedrock Agent")
            error = e
            time.sleep(current_delay)
            retries += 1
            current_delay *= 2

    if response is None:
        logging.error("Failed to invoke Bedrock Agent after %s attempts" % max_retry_attempts)
        raise Exception("Failed to invoke Bedrock Agent after %s attempts. Last error: %s" % (max_retry_attempts, error))

    return response


def invoke_agentcore(
        aws_auth,
        aws_region,
        agent_runtime_arn,
        qualifier,
        user_message,
        conversation_id,
        initial_retry_delay=1, # seconds
        max_retry_attempts=3
):
    """
    Invoke AgentCore agent (retry with exponential backoff).
    API docs: https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agentcore-runtime_InvokeAgent.html
    """

    bedrock_client = boto3.client(
        service_name="bedrock-agentcore",
        region_name=aws_region,
        **aws_auth
    )

    # AgentCore requires conversation_id to be 33 characters long at least
    if len(conversation_id) < 33:
        # Pad
        conversation_id = conversation_id.ljust(33, '0')

    retries = 0
    response = None
    current_delay = initial_retry_delay
    error = None

    while retries <= max_retry_attempts:
        try:
            logging.info("Invoking Bedrock AgentCore with ARN %s, qualifier %s, conversation_id %s" % (agent_runtime_arn, qualifier, conversation_id))
            error = False
            if qualifier is not None:
                response = bedrock_client.invoke_agent_runtime(
                    agentRuntimeArn=agent_runtime_arn,
                    qualifier=qualifier,
                    runtimeSessionId=conversation_id,
                    payload=json.dumps({"prompt": user_message}).encode()
                )
            else:
                response = bedrock_client.invoke_agent_runtime(
                    agentRuntimeArn=agent_runtime_arn,
                    runtimeSessionId=conversation_id,
                    payload=json.dumps({"prompt": user_message}).encode()
                )
            logging.info("Got response from Bedrock AgentCore: %s" % response)
            break

        except Exception as e:
            logging.exception("Error invoking Bedrock AgentCore")
            error = e
            time.sleep(current_delay)
            retries += 1
            current_delay *= 2

    if response is None:
        logging.error("Failed to invoke Bedrock AgentCore after %s attempts" % max_retry_attempts)
        raise Exception("Failed to invoke Bedrock AgentCore after %s attempts. Last error: %s" % (max_retry_attempts, error))

    return response