def extract_return_control_error(event, agent_identifier):
    """
    DSS agents are currently unable to handle human-in-the-loop (HITL) interactions, e.g. when
    an agent attempts to get user confirmation on if a tool should be invoked. If this happens
    with a Bedrock agent, we extract information about the specific tool causing the issue,
    and return this information to the user.

    'event' (partial) schema when attemting to return control to user:

    {
        'returnControl': {
            'invocationInputs': [
                {
                    'functionInvocationInput': {
                        'actionGroup': '<action group name>',
                        'function': '<function name>',
                        ...
                    }
                }
            ],
            ...
        }
    }
    """
    tools_returning_control = \
        {i["functionInvocationInput"]["actionGroup"]: i["functionInvocationInput"]["function"] for i in event["returnControl"]["invocationInputs"]}
    error_msg = f"Error: One or more tools associated with AWS Bedrock Agent {agent_identifier} is requesting " + \
                f"to return control to the user. This is currently not supported by this plugin. Please work with the Bedrock agent owner " + \
                f"to resolve the issue.\n\nThe offending 'Action Group: Functions' are: {str(tools_returning_control)}"
    return error_msg

def extract_kb_source_items(event, kb_source_items):
    """
    Bedrock agents can query internal knowledge banks; when they do, they return the documents
    their response is built on. This function extracts each document, and adds them as a SIMPLE_DOCUMENT
    to the 'kb_source_items' list.

    'event' (partial) schema when kb sources are included:

    {
        'chunk': {
            'attribution': {
                'bytes': b'<agent text response>',
                'citations': [
                    {
                        'retrievedReferences': [
                            {
                                'content': {'text': '<content text>', 'type': 'TEXT'}, 
                                'metadata': {
                                    'x-amz-bedrock-kb-source-uri': '<source uri>', 
                                    ...
                                },
                                ...
                            }
                        ],
                        ...
                    }
                ]
            }
        }
    } 
    """
    if "attribution" in event.get("chunk", {}):
        for citation in event["chunk"]["attribution"]["citations"]:
            for reference in citation["retrievedReferences"]:
                kb_source_items.append(
                    {
                        "type": "SIMPLE_DOCUMENT",
                        "metadata": reference["metadata"],
                        "title": reference["metadata"]["x-amz-bedrock-kb-source-uri"],
                        "textSnippet": reference["content"]["text"]
                    }
                )

def extract_tool_call_sources(event, tool_sources):
    """
    Bedrock agents can leverage internal functions (mainly by calling AWS Lambda functions); related
    functions are grouped into "Actions Groups". When Bedrock calls a function, it emits two 'trace' 
    events: one for the function invocation, and one for the function return.

    This method parses the Bedrock agents 'trace' events, extracts function call inputs and outputs,
    and uses them to populate the metadata of a "source" object. Note that the format of "tool_sources" 
    is different to that of "kb_source_items" (in _extract_kb_source_item), as in this case we need
    to use a dictionary to match function inputs and outputs (i.e. as they are provided in two
    different 'trace' events). Weirdly, the 'eventTime' is the join key.

    'event' (partial) schema when function calls are included:
    {
        'trace': {
            'eventTime': <python datetime.datetime object, wieridly this is the join key>,
            'trace': {
                'orchestrationTrace': {
                    'invocationInput': {
                        'actionGroupInvocationInput': {
                            'actionGroupName': '<action group name>',
                            'function': '<function name>',
                            'parameters': [{
                                'name': '<parameter name>',
                                'type': '<parameter type>',
                                'value': '<parameter value>'
                                },
                                ...
                            ],
                            ...
                        },
                        ...
                    }
                }
            },
            ...
        }
    }

    'event' (partial) schema when function calls returns are included:
    {
        'trace': {
            'eventTime': <python datetime.datetime object, wieridly this is the join key>,
            'trace': {
                'orchestrationTrace': {
                    'observation': {
                        'actionGroupInvocationOutput': {
                            text': '<function output>',
                            ...
                        },
                        ...
                    }
                }
            },
            ...
        }
    }

    """
    if "trace" in event:
        # Extract cunction calls
        if event["trace"]["trace"].get("orchestrationTrace", {}).get("invocationInput", {}).get("actionGroupInvocationInput"):
            action_group_name = event["trace"]["trace"]["orchestrationTrace"]["invocationInput"]["actionGroupInvocationInput"]["actionGroupName"]
            function_name = event["trace"]["trace"]["orchestrationTrace"]["invocationInput"]["actionGroupInvocationInput"]["function"]
            tool_inputs = {p["name"]: p["value"] for p in event["trace"]["trace"]["orchestrationTrace"]["invocationInput"]["actionGroupInvocationInput"]["parameters"]}  

            tool_sources[event["trace"]["eventTime"]] = {
                "toolCallDescription": f"AWS Bedrock Action Group: {action_group_name}",
                "items": [{
                    "type": "SIMPLE_DOCUMENT",
                    "title": f"AWS Bedrock Function: {function_name}",
                    "metadata": {
                        "aws_bedrock_actiongroup": action_group_name,
                        "aws_bedrock_function": function_name,
                        "function_inputs": str(tool_inputs)
                    }
                }]
            }

        # Extract function responses
        if event["trace"]["trace"].get("orchestrationTrace", {}).get("observation", {}).get("actionGroupInvocationOutput"):
            tool_sources[event["trace"]["eventTime"]]["items"][0]["metadata"]["function_outputs"] = \
                event["trace"]["trace"]["orchestrationTrace"]["observation"]["actionGroupInvocationOutput"]["text"]

def process_kb_and_tool_sources(sources, kb_source_items, tool_sources):
    """
    Consolidate 'kb_source_items' and 'tool_sources' into the 'sources' object.
    """
    # Process kb sources
    if kb_source_items:
        sources.append(
            {
                "toolCallDescription": "AWS Bedrock Knowledge Banks",
                "items": kb_source_items
            }
        )

    # Process tool sources
    if tool_sources:
        for _, tool_call in tool_sources.items():
            sources.append(tool_call) 