from dataiku.code_studio import get_dataiku_user_uid

DOCKERFILE = """
##### OPENCODE BLOCK #####

USER root
RUN dnf module install nodejs:22/common -y && \
    npm install -g opencode-ai@{opencode_version}

# Add a README to the Code Studio Workspace
USER {uid}
COPY --chown=dataiku:dataiku ./OpenCode_README.md /home/dataiku/workspace
"""

VSCODE_EXT_DOCKERFILE = """
# Add VS code extension
USER {uid}
RUN code-server \
    --extensions-dir /home/dataiku/.local/share/code-server/local-extensions \
    --install-extension {vscode_ext}
"""

OPENCODE_README = """# OpenCode_README.md

## Using the OpenCode AI Assistant

This Code Studio is equipped with the OpenCode AI assistant to help with your development tasks. You can interact with it through its Terminal User Interface (TUI), the command-line interface (CLI), or the integrated VS Code extension.

### Terminal User Interface (TUI)

To start the interactive TUI, open a new terminal and run:
```bash
opencode
```

Inside the TUI, you can:
- **Reference files:** Use `@` to include files in your prompt (e.g., `How is auth handled in @src/api/index.js?`).
- **Run shell commands:** Start your message with `!` to execute a shell command (e.g., `!ls -la`).
- **Use slash commands:** Type `/` to see a list of available commands like `/help`, `/new`, or `/sessions`.

### Command-Line Interface (CLI)

To use the assistant directly from the command line without entering the TUI, use the `opencode run` command followed by your request in natural language.

For example:
```bash
opencode run "write a python function that reads a CSV file with pandas and returns a dataframe"
```

### VS Code Companion Extension

A companion VS Code extension for OpenCode can also be installed. This provides a more integrated experience, allowing for features such as inline code generation and a dedicated chat interface.

To activate the extension, open the Command Palette (`F1`) and search for "OpenCode".

### Assistant Guidance (`AGENTS.md`)

An `AGENTS.md` file has been included in **code_studio-resources**. This file contains important context and instructions for the AI assistant, tailored to the Dataiku Code Studio environment. You can review and modify this file to further customize the assistant's behavior.

"""

AGENTS_MD = """# AGENTS.md

This file provides guidance to AI coding assistants when working with code in this repository.

## Project Structure
This is a Dataiku Code Studio workspace that provides a development environment integrated with Dataiku DSS. 
The workspace is organized into synchronized folders that connect to different parts of the Dataiku platform:

### Synchronized Directories
- **notebooks/**: Jupyter notebooks for the project (Python, R, Scala). Only create notebooks at root level - subfolders are ignored
- **recipes/**: Contains coding recipes from the project flow. Files can only be edited, not created (recipes must be created in DSS)
- **code_studio-versioned/**: Files specific to this Code Studio, synchronized to DSS versioned storage
- **project-lib-versioned/**: Project library content, accessible through DSS Libraries
- **user-versioned/**: User-specific settings synchronized across all Code Studios

### Non-versioned Resources
- **code_studio-resources/**: Code Studio-specific static files and artifacts
- **project-lib-resources/**: Shared project resources (non-versioned)
- **user-resources/**: User-specific resources synchronized across Code Studios

## Development Environment
- This is a containerized VS Code environment running in Kubernetes
- Primary language support: Python (dataiku library available)
- No traditional package managers (pip, npm, etc.) - dependencies managed through Dataiku
- Jupyter notebooks are the primary development interface

## Code Environment Discovery and Selection
To ensure correct execution context, the assistant must identify and select the required Python code environment:

**Trigger:** Whenever the user asks to execute, run, or test any **Python** code, the assistant **must** first perform the following steps to ensure the correct Python execution context.

1.  **Discovery:** Dataiku default Python code environment, with Dataiku library available, is available at /opt/dataiku/pyenv. Other Python code environments with Dataiku library available reside in the following directory: /opt/dataiku/python-code-envs
    * If **multiple** code environments are found in this directory, the assistant **must** list them and prompt the user to select the desired environment before proceeding with any code-related task.
    * If only **one** code environment is found, the assistant may automatically proceed using that environment.

## Working with Dataiku
- Standard imports for Dataiku development
```python
# Standard imports for Dataiku development
import dataiku
from dataiku import pandasutils as pdu
import pandas as pd
```

## Important Constraints
⚠️ **Critical**: Files stored at the workspace root are NOT synchronized with DSS and will be lost when the Code Studio stops. 
                Always use the appropriate synchronized subfolders.
⚠️ **Notebooks**: Only create notebooks directly in the `notebooks/` folder - subfolders are ignored by Dataiku.
⚠️ **Recipes**: The `recipes/` folder is read-only for file creation - recipes must be created through the DSS interface.

## Development Workflow
1. Use `notebooks/` for exploratory data analysis and prototyping
2. Store reusable code in `code_studio-versioned/` or `project-lib-versioned/`
3. Keep static resources in appropriate `*-resources/` folders
4. Access Dataiku datasets and APIs through the `dataiku` Python library
"""


def dockerfile(
    oc_version="latest",
    add_vscode_ext=False,
    vscode_ext_version=""
):
    """
    Returns Dockerfile to install OpenCode coding assistant.
    
    If `add_md`, adds a RUN statement that copies `md_filename`
    from the docker build env to the root of the Code Studio workspace,
    i.e. /home/dataiku/workspace.
    
    If `add_vscode_plugin`, adds a RUN statement that installs the 
    VS Code companion extension to the CLI assistant.
    
    Args:
      add_md: bool
      md_filename: filename for agent markdown file
      add_vscode_ext: bool
      vscode_ext_version: vscode extension version
    
    Returns:
      A dockerfile snippet to install the requested assistant.
    """
    # Compose dockerfile
    dataiku_uid = get_dataiku_user_uid()
    dockerfile = DOCKERFILE.format(uid=dataiku_uid, opencode_version=oc_version if oc_version else "latest")
    
    if add_vscode_ext:
        vscode_ext = "sst-dev.opencode"
        # If extension version is provided, install the specific vscode extension version
        if vscode_ext_version:
            vscode_ext = vscode_ext + f"@{vscode_ext_version}"
        dockerfile += VSCODE_EXT_DOCKERFILE.format(uid=dataiku_uid, vscode_ext=vscode_ext)
        
    return dockerfile


def get_md_file():
    """
    Returns the agent markdown file.
    """
    return AGENTS_MD
    

def get_readme_file():
    """
    Returns the README markdown file.
    """
    return OPENCODE_README
    