def create_configmap(name, namespace, key, value):
    """
    Return a Kubernetes ConfigMap as PyYAML-compliant dictionary.
    """
    configmap_json = {
        "apiVersion": "v1",
        "kind": "ConfigMap",
        "metadata": {
            "name": name,
            "namespace": namespace
        },
        "data": {
            key: value
        }
    }
    return configmap_json
    

def add_volume_to_deployment(deployment, volume):
    """
    Adds a volume to the deployment.
    
    Args:
        deployment (dict): Code Studios Deployment as PyYAML-compliant dictionary
        volume (str): The volume definition as PyYAML-compliant dictionary
    """
    if not deployment["spec"]["template"]["spec"].get("volumes"):
        deployment["spec"]["template"]["spec"]["volumes"] = []

    deployment["spec"]["template"]["spec"]["volumes"].append(volume)
    

def mount_volume_in_all_containers(deployment, volume_name, volume_mount_path):
    """
    Mounts a volume in all containers of the deployment.
    
    Args:
        deployment (dict): Code Studios Deployment as PyYAML-compliant dictionary
        volume_name (str): The name of the volume to mount
        volume_mount_path (str): The path to mount the volume at
    """
    for container in deployment["spec"]["template"]["spec"]["containers"]:
        if not container.get("volumeMounts"):
            container["volumeMounts"] = []

        container["volumeMounts"].append({
            "name": volume_name,
            "mountPath": volume_mount_path
        })


def add_env_var_with_value_to_deployment(
    deployment,
    env_var_name,
    env_var_value
):
    """
    Adds an environment variable with a static value to the containers of a Deployment.
    
    Args:
      deployment: Code Studios Deployment as PyYAML-compliant dictionary
      env_var_name: environment variable name
      env_var_value: value of environment variable
    
    Returns:
      None (Deployment modified in-place)
    """
    env_var = {
        'name': env_var_name,
        'value': env_var_value
    }
    for container in deployment["spec"]["template"]["spec"]["containers"]:
        if "env" not in container:
            container["env"] = []
        container["env"].append(env_var)


def add_post_start_hook_to_deployment(deployment, command):
    """
    Adds the `command` as a posStart hook to the `deployment`.
    
    Args:
      deployment: Code Studios Deployment as PyYAML-compliant dictionary
      command: command(s) to be added, as a list
    
    Returns:
      None (Deployment modified in-place)
    """
    for container in deployment["spec"]["template"]["spec"]["containers"]:
        container["lifecycle"] = {
            "postStart": {
                "exec": {
                    "command": command
                }
            }
        }


def get_env_var_from_container_deployment(deployment, container_name="c", env_var_name=""):
    """
    Returns the env var object from a container spec.
    """
    envs = []
    for container in deployment["spec"]["template"]["spec"]["containers"]:
        if container["name"] == container_name:
            envs = container.get("env", [])

    return next((env for env in envs if env["name"] == env_var_name), None)
