import { expect } from '@playwright/test';
import { test } from "../../../playwright.setup";

const loadPage = async (page, baseURL) => {
  expect(baseURL).toBeTruthy();
  await page.goto(baseURL!);
  await expect(
    page.locator("div").filter({ hasText: "Loading data..." }).nth(2)
  ).not.toBeVisible();
  await expect(page.locator(".q-spinner")).toHaveCount(0);
};


// Helper function to verify dataset selection state
async function verifyDatasetSelection(page, expectedDisplayName = null) {
  const selectionState = await page.evaluate(() => {
    try {
      // Find the dataset selector by its input element
      const inputElement = document.querySelector('input[ng-model="datasetFilter"]');
      if (!inputElement) return { error: 'Dataset filter input not found' };

      const scope = angular.element(inputElement).scope();
      const controllerScope = angular.element(document.querySelector('[ng-controller="LLMConfigController"]')).scope();

      const configProp = scope.configProperty;

      return {
        isDropdownOpen: scope.isDatasetDropdownOpen,
        configProperty: configProp,
        configValue: controllerScope.config[configProp],
        selectedDataset: scope.selectedDataset,
        datasetFilter: scope.datasetFilter,
        isValid: controllerScope.config[configProp] === (scope.selectedDataset ? scope.selectedDataset.datasetName : null),
        availableDatasets: scope.datasets.length,
        datasetsByType: Object.keys(scope.datasetsByType || {})
      };
    } catch (e) {
      return { error: e.message, stack: e.stack };
    }
  });

  console.log('Dataset Selection State:', JSON.stringify(selectionState, null, 2));

  if (selectionState.error) {
    throw new Error(`Failed to get dataset selection state: ${selectionState.error}`);
  }

  return selectionState;
}

// Helper function to get available datasets in dropdown
async function getAvailableDatasets(page) {
  return await page.evaluate(() => {
    const datasets = [];
    const datasetElements = document.querySelectorAll('div[ng-repeat*="dataset in datasets"][ng-if*="datasetMatchesFilter"]');

    datasetElements.forEach(element => {
      datasets.push(element.textContent.trim());
    });

    return datasets;
  });
}

test('Dataset Selector - Complete functionality test', async ({ page, baseURL }) => {
  // Navigate and setup
  await loadPage(page, baseURL);

  // Wait for page to load
  await page.waitForSelector('[ng-controller="LLMConfigController"]', {
    state: 'visible',
    timeout: 15000
  });
  await page.waitForLoadState('networkidle');

  // Navigate to the section containing dataset selector
  await page.click('a[href="#llmConfig"]');
  await page.waitForSelector('#section-llmConfig', { state: 'visible' });

  // Wait for dataset selector to be available
  await page.waitForSelector('input[ng-model="datasetFilter"]', {
    timeout: 10000
  });

  console.log('=== Test 1: Click dropdown and verify it opens ===');


  // Click on the dropdown arrow to open
  await page.locator('dataset-selector')
    .filter({ hasText: 'Select history dataset' })
    .locator('.dku-icon-chevron-down-16')
    .click({ timeout: 5000 });


  // Wait for dropdown to open
  await page.waitForSelector('div[ng-if="isDatasetDropdownOpen"]', {
    state: 'visible',
    timeout: 10000
  });

  // Verify dropdown is open
  let state = await verifyDatasetSelection(page);
  expect(state.isDropdownOpen).toBe(true);
  expect(state.availableDatasets).toBeGreaterThan(0);

  // Wait for dataset options to load
  await page.waitForSelector('div[ng-repeat*="dataset in datasets"]', {
    timeout: 10000
  });

  console.log('✅ Dropdown opened successfully');

  console.log('=== Test 2: Click clear button ===');

  // First, let's select something so we have something to clear
  await page.locator('div[ng-repeat*="dataset in datasets"][ng-click*="selectDataset"]')
    .first()
    .click({ timeout: 10000 });

  // Wait for selection to process
  await page.waitForTimeout(1000);

  // Verify something is selected
  state = await verifyDatasetSelection(page);
  expect(state.selectedDataset).toBeTruthy();
  expect(state.datasetFilter).toBeTruthy();
  console.log('Dataset selected for clearing test:', state.datasetFilter);

  //Now click the clear button
  await page.locator('dataset-selector')
  .filter({ hasText: 'Select history dataset' })
  .locator('.dku-icon-dismiss-16')
  .click({ timeout: 5000 });



  // Wait for clearing to process
  await page.waitForTimeout(500);

  // Verify selection is cleared
  state = await verifyDatasetSelection(page);
  expect(state.selectedDataset).toBeNull();
  expect(state.datasetFilter).toBe('');
  expect(state.configValue).toBeNull();

  console.log('✅ Clear button worked successfully');

  console.log('=== Test 3: Select first dataset and verify config ===');

  // The dropdown should still be open after clearing (based on clearSelection function)
  // If not, open it again
  const isDropdownVisible = await page.isVisible('div[ng-if="isDatasetDropdownOpen"]');
  if (!isDropdownVisible) {
    await page.locator('.dku-icon-chevron-down-16').click();
    await page.waitForSelector('div[ng-if="isDatasetDropdownOpen"]', { state: 'visible' });
  }

  // Get available datasets for logging
  const availableDatasets = await getAvailableDatasets(page);
  console.log('Available datasets:', availableDatasets);

  // Click the first dataset option (excluding type headers)
  await page.locator('div[ng-repeat*="dataset in datasets"][ng-if*="datasetMatchesFilter"]')
    .first()
    .click({ timeout: 10000 });

  // Wait for selection to process
  await page.waitForTimeout(1000);

  // Verify the final selection
  state = await verifyDatasetSelection(page);

  console.log('Final selection state:', {
    configValue: state.configValue,
    selectedDatasetName: state.selectedDataset?.datasetName,
    datasetFilter: state.datasetFilter,
    isValid: state.isValid
  });

  // Main assertions
  expect(state.selectedDataset).toBeTruthy();
  expect(state.selectedDataset.datasetName).toBeTruthy();
  expect(state.configValue).toBe(state.selectedDataset.datasetName);
  expect(state.datasetFilter).toBe(state.selectedDataset.datasetDisplayName);
  expect(state.isValid).toBe(true);
  expect(state.isDropdownOpen).toBe(false); // Should close after selection

  console.log('✅ Dataset selection and config verification successful');
  console.log(`Selected dataset: ${state.selectedDataset.datasetName}`);
});
