from datetime import datetime
from typing import Optional

from answers.backend.db.sql.dataset_schemas import ANSWERS_DATASETS_METADATA, GENERAL_FEEDBACK_DATASET_CONF_ID
from common.backend.db.sql.queries import InsertQueryBuilder, get_post_queries
from common.backend.db.sql.tables_managers import GenericFeedbackSQL
from common.backend.utils.dataiku_api import dataiku_api
from common.backend.utils.sql_timing import log_query_time


class GeneralFeedbackSQL(GenericFeedbackSQL):
    def __init__(self):
        config = dataiku_api.webapp_config
        if not config.get("allow_general_feedback", False):
            return
        super().__init__(config=config,
                         columns=ANSWERS_DATASETS_METADATA[GENERAL_FEEDBACK_DATASET_CONF_ID]["columns"],
                         dataset_conf_id=GENERAL_FEEDBACK_DATASET_CONF_ID,
                         default_project_key=dataiku_api.default_project_key)

    @log_query_time
    def add_feedback(
        self,
        timestamp: datetime,
        user: str,
        message: str,
        knowledge_bank_id: Optional[str] = None,
        llm_id: Optional[str] = None,
    ):
        record_value = [
            timestamp.isoformat(),
            user,
            message,
            knowledge_bank_id if knowledge_bank_id is not None else "",
            llm_id if llm_id is not None else "",
        ]
        insert_query = (
            InsertQueryBuilder(self.dataset)
            .add_columns(self.columns)
            .add_values(values=[record_value])
            .build()
        )
        self.executor.query_to_df(insert_query, post_queries=get_post_queries(self.dataset))


feedbacks_sql_manager = GeneralFeedbackSQL()
