
import io
import unittest

from common.backend.utils.file_utils import get_file_mime_type, is_file_coherent
from sample_file_content import (
    CORRUPTED_MD_HEX,
    EXTENDED_MD,
    MINIMAL_DOCX,
    MINIMAL_MD,
    MINIMAL_PDF,
    MINIMAL_PNG,
    MINIMAL_PPTX,
)
from werkzeug.datastructures import FileStorage


class TestFileUtils(unittest.TestCase):
    def test_get_file_mime_type(self):
        test_cases = [
            # images
            ("image.png", "image/png"),
            ("photo.jpeg", "image/jpeg"),
            ("graphic.webp", "image/webp"),
            ("animation.gif", "image/gif"),
            
            # documents
            ("document.pdf", "application/pdf"),
            ("report.docx", "application/vnd.openxmlformats-officedocument.wordprocessingml.document"),
            ("notes.txt", "text/plain"),
            ("data.json", "application/json"),
            ("script.py", "text/x-python"),
            ("page.html", "text/html"),
            ("app.js", "text/javascript"),
            ("readme.md", "text/markdown"),

            # weird things
            ("file_with_no_extension", None), 
            ("strange.name.with.many.dots.tar.gz", "application/x-tar"), 
            ("UPPERCASE.TXT", "text/plain"), 
            ("weird&name@file!.html", "text/html"), 
        ]

        # Collect failed cases to report all MIME type mismatches at once.

        failed_cases = []

        for file_name, expected_mime_type in test_cases:
            mime_type = get_file_mime_type(file_name)
            if mime_type != expected_mime_type:
                failed_cases.append((file_name, mime_type, expected_mime_type))

        if failed_cases:
            error_messages = [
                f"File '{file_name}' returned MIME type '{mime_type}' but expected '{expected_mime_type}'."
                for file_name, mime_type, expected_mime_type in failed_cases
            ]
            self.fail("\n".join(error_messages))

    def test_is_file_coherent(self):
        def make_file_storage(data: bytes, filename: str) -> FileStorage:
            """
            Helper to create a simulated FileStorage
            """
            return FileStorage(
                stream=io.BytesIO(data),
                filename=filename,
                content_type='application/octet-stream'
            )

        CORRUPTED_MD = bytes.fromhex(CORRUPTED_MD_HEX.replace("\n", "").replace(" ", ""))

        test_cases = [
            (None, False),  # File is None
            (
                # Valid TXT content with correct extension
                make_file_storage(b"file content" * 100, "test.txt"), 
                True
            ),  
            (
                # Valid PNG content with wrong extension
                make_file_storage(MINIMAL_PNG, "image.pdf"),
                False,
            ),  
            (
                # Valid PNG content with correct extension
                make_file_storage(MINIMAL_PNG, "test.png"),
                True,
            ),  
            (
                # Valid PDF content with correct extension
                make_file_storage(MINIMAL_PDF, "test.pdf"),
                True,
            ),
            (
                # Valid PPTX content with correct extension
                make_file_storage(MINIMAL_PPTX, "test.pptx"),
                True,
            ),
            (
                # Valid DOCX content with correct extension
                make_file_storage(MINIMAL_DOCX, "my.test.docx"),
                True,
            ),
            (
                # Valid MD content with correct extension
                make_file_storage(MINIMAL_MD, "minimal.md"),
                True,
            ),
            (
                # Valid MD content with correct extension
                make_file_storage(EXTENDED_MD, "extended.md"),
                True,
            ),
            (
                # MD file with binary content
                make_file_storage(CORRUPTED_MD, "corrupted.md"),
                False,
            ),
            (FileStorage(stream=io.BytesIO(b"data"), filename=None), False),  # File with no filename
        ]

        for file, expected in test_cases:
            result = is_file_coherent(file)
            assert result == expected, f"Expected {expected} for file {file.filename if file else 'None'}, got {result}"